import sandbox.sdk2 as sdk2
import logging
from itertools import islice

MAX_ROWS_PER_TRANSACTION = 40000

TABLES_NAME = ['traffic_types', 'segments', 'logins', 'domains', 'pages', 'dsps']

DROP_SUGGEST_QUERY = '''
PRAGMA yt.TmpFolder = '//home/yabs/tac-manager/tmp';

DROP TABLE `//home/yabs/tac-manager/suggest/traffic_types`;
DROP TABLE `//home/yabs/tac-manager/suggest/segments`;
DROP TABLE `//home/yabs/tac-manager/suggest/logins`;
DROP TABLE `//home/yabs/tac-manager/suggest/domains`;
DROP TABLE `//home/yabs/tac-manager/suggest/pages`;
DROP TABLE `//home/yabs/tac-manager/suggest/dsps`;
'''

INSERT_SUGGEST_QUERY = '''
PRAGMA yt.TmpFolder = '//home/yabs/tac-manager/tmp';

$segment_traffic_type_table = "//home/comdep-analytics/YAN/segmentation/static/latest/blocks";

INSERT INTO `//home/yabs/tac-manager/suggest/traffic_types`
SELECT
    DISTINCT traffic_type AS Name
FROM $segment_traffic_type_table;

INSERT INTO `//home/yabs/tac-manager/suggest/segments`
SELECT
    DISTINCT segment AS Name
FROM $segment_traffic_type_table;

$partner_page_table = "//home/yabs/dict/PartnerPage";

INSERT INTO `//home/yabs/tac-manager/suggest/logins`
SELECT
    DISTINCT Login AS Name
FROM $partner_page_table;

INSERT INTO `//home/yabs/tac-manager/suggest/domains`
SELECT
    DISTINCT (String::SplitToList(DomainList, ",")[0]) AS Name
FROM $partner_page_table;

INSERT INTO `//home/yabs/tac-manager/suggest/pages`
SELECT
    Name,
    Description,
    PageID
FROM `//home/yabs/dict/Page`;

INSERT INTO `//home/yabs/tac-manager/suggest/dsps`
SELECT
    Title,
    Hostname,
    Path,
    DSPID
FROM `//home/yabs/dict/DSPTemplate`;
'''


def gen_chunks(gen, chunk_size):
    it = iter(gen)
    while True:
        chunk_slice = islice(it, chunk_size)
        chunk = [row for row in chunk_slice]
        if not chunk:
            break
        yield chunk


class TacmanUpdateSuggestTables(sdk2.Task):
    '''Update suggest tables'''

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('YT parameters') as yt_parameters:
            yt_vault_token = sdk2.parameters.String(
                'Your yt token name in vault',
                default='yt-tacman-token',
                required=True
            )

            yql_vault_token = sdk2.parameters.String(
                'Your yql token name in vault',
                default='yql-tacman-token',
                required=True
            )

            yt_host = sdk2.parameters.String(
                'yt proxy',
                default='hahn',
            )

    def on_execute(self):
        from yt.wrapper import YtClient
        from yql.api.v1.client import YqlClient

        yql_client = YqlClient(
            db=self.Parameters.yt_host,
            token=sdk2.Vault.data(self.owner, self.Parameters.yql_vault_token),
        )
        yt = YtClient(
            token=sdk2.Vault.data(self.owner, self.Parameters.yql_vault_token),
            proxy=self.Parameters.yt_host,
        )
        request = yql_client.query(
            DROP_SUGGEST_QUERY
        )
        request.run()
        request.wait_progress()
        results = request.get_results()
        if not results.is_success:
            msg = "\n".join([str(err) for err in results.errors])
            raise Exception('Error when executing DROP TABLES query: {}'.format(msg))

        request = yql_client.query(
            INSERT_SUGGEST_QUERY
        )
        request.run()
        request.wait_progress()
        results = request.get_results()
        if not results.is_success:
            msg = "\n".join([str(err) for err in results.errors])
            raise Exception('Error when executing DROP TABLES query: {}'.format(msg))

        for table in TABLES_NAME:
            static_table_path = '//home/yabs/tac-manager/suggest/' + table
            dyn_table_path = '//home/yabs/tac-manager/suggest/DYN_' + table
            schema = yt.get_attribute(dyn_table_path, 'schema')
            logging.debug('Remove {} table'.format(dyn_table_path))
            yt.remove(dyn_table_path)
            yt.create("table", dyn_table_path, attributes={"schema": schema, 'dynamic': True})
            yt.mount_table(dyn_table_path)
            logging.debug('Copy from {} to {}'.format(static_table_path, dyn_table_path))
            chunk_id = 0
            rows = yt.read_table(static_table_path)
            for chunk in gen_chunks(rows, MAX_ROWS_PER_TRANSACTION):
                yt.insert_rows(dyn_table_path, chunk)
                chunk_id += 1
