import calendar
import datetime
import logging
import sandbox.sdk2 as sdk2


TABLE_PATH = '//home/yabs/tac-manager/turnover/turnover_{year}_{month}'

LAST_TABLE_LINK = '//home/yabs/tac-manager/turnover/last'

INSERT_TURNOVER_QUERY = '''
USE hahn;
PRAGMA SimpleColumns;
PRAGMA ClassicDivision = 'False';
PRAGMA yt.PoolTrees = 'physical';
PRAGMA yt.TentativePoolTrees = 'cloud';
-- PRAGMA yt.AutoMerge = 'disabled';
-- PRAGMA yson.DisableStrict;
PRAGMA yt.TmpFolder = '//home/yabs/tac-manager/tmp';
PRAGMA AnsiInForEmptyOrNullableItemsCollections;
PRAGMA yt.Pool = 'ads-research';

$date_from = '{year}-{month}-01';
$date_to   = '{year}-{month}-{end_day}';

INSERT INTO `//home/yabs/tac-manager/turnover/turnover_{year}_{month}`

SELECT
    CAST(page_id as UINT64) as pageid,
    CAST(block_id as UINT64) as impid,
    dsp_id as dspid,

    SUM(NVL(rtb_slots_sold, 0)) as slots_sold,
    SUM(NVL(rtb_slots_shown, 0)) as slots_shown,

    SUM(NVL(rtb_revenue, 0)) as rtb_revenue,
    SUM(NVL(cube_revenue, 0)) as revenue,
    SUM(NVL(cube_revenue_w_cpa, 0)) as revenue_w_cpa,
    NVL(SUM_IF(cube_revenue_w_cpa,NVL(commerce_type, '') != 'non-commerce'), 0.0)  as revenue_w_cpa_nc,

    SUM(NVL(cube_tac, 0)) as tac,
    SUM(NVL(rtb_tac, 0)) as rtb_tac,

    SUM(NVL(direct_ad_shows, 0)) as direct_ad_shows,
    SUM(NVL(direct_ad_clicks, 0)) as direct_ad_clicks
FROM `//home/comdep-analytics/YAN/cubes/yan_new/latest/views/seg-latest/materialized`
WHERE
    report_date >= $date_from AND
    report_date <= $date_to AND
    page_id IS NOT NULL AND
    dsp_id IS NOT NULL AND
    network_type == 'yan' AND
    ownership_type == 'external' AND
    NVL(fraud_type, '') NOT IN ('flagged', 'fraud')
    -- NVL(commerce_type, '') != 'non-commerce' AND
GROUP BY
    page_id as page_id,
    NVL(block_id, 0) as block_id,
    dsp_id as dsp_id
HAVING
    SUM(NVL(rtb_slots_sold, 0)) > 0 OR
    SUM(NVL(rtb_slots_shown, 0)) > 0 OR
    SUM(NVL(rtb_revenue, 0)) > 0 OR
    SUM(NVL(cube_revenue, 0)) > 0 OR
    SUM(NVL(cube_revenue_w_cpa, 0)) > 0 OR
    SUM(NVL(cube_tac, 0)) > 0 OR
    SUM(NVL(rtb_tac, 0)) > 0 OR
    SUM(NVL(direct_ad_shows, 0)) > 0 OR
    SUM(NVL(direct_ad_clicks, 0)) > 0
ORDER BY
    pageid,
    impid,
    dspid
;
'''


class TacmanUpdateTurnoverTables(sdk2.Task):
    '''Update turnover tables'''

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('YT parameters') as yt_parameters:
            yt_vault_token = sdk2.parameters.String(
                'Your yt token name in vault',
                default='yt-tacman-token',
                required=True
            )

            yql_vault_token = sdk2.parameters.String(
                'Your yql token name in vault',
                default='yql-tacman-token',
                required=True
            )

            yt_host = sdk2.parameters.String(
                'yt proxy',
                default='hahn',
            )

    def on_execute(self):
        from yt.wrapper import YtClient
        from yql.api.v1.client import YqlClient

        yql_client = YqlClient(
            db=self.Parameters.yt_host,
            token=sdk2.Vault.data(self.owner, self.Parameters.yql_vault_token),
        )
        yt = YtClient(
            token=sdk2.Vault.data(self.owner, self.Parameters.yql_vault_token),
            proxy=self.Parameters.yt_host,
        )
        today = datetime.datetime.today()
        first = today.replace(day=1)
        lastMonth = first - datetime.timedelta(days=1)
        end_day = calendar.monthrange(lastMonth.year, lastMonth.month)[1]
        month = lastMonth.strftime("%m")
        year = lastMonth.strftime("%Y")
        request = yql_client.query(
            INSERT_TURNOVER_QUERY.format(month=month, year=year, end_day=end_day)
        )
        logging.debug('Get results from range date_from {year}-{month}-1 to {year}-{month}-{end_day};'.format(
            year=year, month=month, end_day=end_day
        ))
        request.run()
        request.wait_progress()
        results = request.get_results()
        if not results.is_success:
            msg = "\n".join([str(err) for err in results.errors])
            raise Exception('Error when executing INSERT TABLES query: {}'.format(msg))
        yt.link(TABLE_PATH.format(month=month, year=year), LAST_TABLE_LINK, force=True)
