import subprocess
from six import iteritems
from datetime import datetime
from sandbox.projects.yabs.partner_share.lib.config.config import (
    get_config,
    get_daemons,
)

config = get_config()
daemons = get_daemons(config)


def make_width(chars):
    return round(chars * 0.13 + 0.3, 2)


def make_width_low(chars):
    return round(chars * 0.1 + 0.3, 2)


def build_states():
    states = {}
    for stage_name in config['stages']:
        stage = config['stages'][stage_name]
        if 'performer' in stage:
            name = f'"{stage_name}\\n({stage["performer"]})"'
            length = max(len(stage_name), len(stage['performer']))
        else:
            name = stage_name
            length = len(stage_name)
        states[stage_name] = dict(
            name=name,
            descr=stage['description'],
            width=make_width(length),
            is_stage=True,
            performer=stage.get('performer'),
            color='#77aaff',
            url=f'https://cs.yandex-team.ru/#!%22{stage_name}%22%3A%20%7B,partner_share%2F.*%2Fconfig.json,m,arcadia,,5000',
            style='bold,rounded,filled',
        )
        if 'next_states' in stage:
            if 'success' in stage['next_states']:
                success_name = stage["next_states"]["success"]
                if success_name not in states:
                    states[success_name] = dict(
                        name=success_name,
                        descr='',
                        width=make_width(len(success_name)),
                        is_stage=False,
                        performer='',
                        color='#77ff77',
                        url='',
                    )
            if 'fail' in stage['next_states'] and stage['next_states']['success'] != stage['next_states']['fail']:
                fail_name = stage['next_states']['fail']
                if fail_name not in states:
                    states[fail_name] = dict(
                        name=fail_name,
                        descr='',
                        width=make_width(len(fail_name)),
                        is_stage=False,
                        performer='',
                        color='#ff7777',
                        url='',
                    )
    return states


def generate_stages_gv():
    r = []
    r.append('digraph TacMan {')
    curdate = datetime.today().strftime('%Y-%m-%d')
    r.append('    label="Stages graph\\n(rendered {})";'.format(curdate))
    r.append('    compound=true;')
    r.append('    style=bold')
    r.append('    fontsize=18')
    r.append('    labelloc=t')
    r.append('    rankdir=TD;')
    r.append('    size="8";')

    r.append('    node  [style="rounded,filled", shape=box, fixedsize=true, width=1.6, fontname="Arial"];')
    states = build_states()
    for state_name, state in iteritems(states):
        width = make_width(len(state_name))
        r.append(f'    {state_name} [tooltip="{state["descr"]}", href="{state["url"]}", width={width}, fillcolor="{state["color"]}"]')

    for stage_name in config['stages']:
        stage = config['stages'][stage_name]
        if 'allowed_previous_states' in stage:
            for prev_name in stage['allowed_previous_states']:
                r.append(f'{prev_name} -> {stage_name}')
        if 'next_states' in stage:
            if 'success' in stage['next_states']:
                success_name = stage["next_states"]["success"]
                r.append(f'    {stage_name} -> {success_name}')
                if 'delay_next_stage_on_success' in stage:
                    next_name = stage['delay_next_stage_on_success']['next_stage']
                    r.append(f'    {success_name} -> {next_name}')
            if 'fail' in stage['next_states'] and stage['next_states']['success'] != stage['next_states']['fail']:
                r.append(f'    {stage_name} -> {stage["next_states"]["fail"]}')
    r.append('}')
    return r


def generate_stages_tasks_gv():
    r = []
    r.append('digraph TacMan {')
    curdate = datetime.today().strftime('%Y-%m-%d')
    r.append('    label="Stages graph (with tasks, which perform stages)\\n(rendered {})";'.format(curdate))
    r.append('    compound=true;')
    r.append('    style=bold')
    r.append('    fontsize=18')
    r.append('    labelloc=t')
    r.append('    rankdir=TD;')
    r.append('    size="8";')

    r.append('    node  [style="rounded,filled", shape=box, fixedsize=true, width=1.6, fontname="Arial"];')
    states = build_states()
    for state_name, state in iteritems(states):
        r.append(f'    {state_name} [label={state["name"]}, tooltip="{state["descr"]}", href="{state["url"]}", width={state["width"]}, fillcolor="{state["color"]}"]')

    for stage_name in config['stages']:
        stage = config['stages'][stage_name]
        if 'allowed_previous_states' in stage:
            for prev_name in stage['allowed_previous_states']:
                r.append(f'{prev_name} -> {stage_name}')
        if 'next_states' in stage:
            if 'success' in stage['next_states']:
                success_name = stage["next_states"]["success"]
                r.append(f'    {stage_name} -> {success_name}')
                if 'delay_next_stage_on_success' in stage:
                    next_name = stage['delay_next_stage_on_success']['next_stage']
                    r.append(f'    {success_name} -> {next_name}')
            if 'fail' in stage['next_states'] and stage['next_states']['success'] != stage['next_states']['fail']:
                r.append(f'    {stage_name} -> {stage["next_states"]["fail"]}')
    r.append('}')
    return r


def generate_ops():
    r = []
    r.append('{')
    r.append('    rank=same')
    prev_op = None
    for op_name in config['stages']['TEST']['operations']:
        op = config['operations'][op_name]
        width = make_width_low(len(op_name))
        url=f'https://cs.yandex-team.ru/#!%22{op_name}%22%3A%20%7B,partner_share%2F.*%2Fconfig.json,m,arcadia,,5000'
        if op.get('memoize_stage'):
            color = 'white'
        else:
            color = '#ccaaee'
        r.append(f'    {op_name} [tooltip="{op["title"]}", href="{url}", width={width}, fillcolor="{color}"]')
        if prev_op:
            r.append(f'    {prev_op} -> {op_name}')
        prev_op = op_name
    r.append('}')
    return r


def generate_ops_block():
    r = []
    r.append('subgraph cluster_departments {')
    r.append('    style=filled;')
    r.append('    color=lightgrey;')
    r.append('    fontsize=18')
    r.append('    label="Operations"')
    r.append('    width=10')
    r.append('    fixedsize=true')
    r.append('{')
    r.append('    rank=same')
    prev_op = None
    for op_name in config['stages']['TEST']['operations']:
        op = config['operations'][op_name]
        width = make_width_low(len(op_name))
        url=f'https://cs.yandex-team.ru/#!%22{op_name}%22%3A%20%7B,partner_share%2F.*%2Fconfig.json,m,arcadia,,5000'
        if op.get('memoize_stage'):
            color = 'white'
        else:
            color = '#ccaaee'
        r.append(f'    {op_name} [tooltip="{op["title"]}", href="{url}", width={width}, fillcolor="{color}"]')
        if prev_op:
            r.append(f'    {op_name} -> {prev_op} [dir=back]')
        prev_op = op_name
    r.append('}')
    r.append('}')
    return r


def generate_operations_gv():
    r = []
    r.append('digraph TacMan {')
    curdate = datetime.today().strftime('%Y-%m-%d')
    r.append('    label="Operations by stages (blue)\\n(rendered {})";'.format(curdate))
    r.append('    compound=true;')
    r.append('    style=bold')
    r.append('    fontsize=18')
    r.append('    labelloc=t')
    r.append('    rankdir=LR;')
    r.append('    size="8";')

    r.append('    node  [style="bold,rounded,filled", shape=box, fixedsize=true, width=1.6, fontname="Arial"];')

    states = build_states()

    for stage_name in config['stages']:
        stage = config['stages'][stage_name]
        state = states[stage_name]
        if 'operations' not in stage:
            continue

        width = make_width(len(stage_name))
        r.append(f'    {stage_name} [tooltip="{state["descr"]}", href="{state["url"]}", width={width}, fillcolor="{state["color"]}"]')

        for op_name in stage['operations']:
            if stage_name in ('TEST', 'FILTER'):
                r.append(f'    {op_name} -> {stage_name} [dir=none]')
            else:
                r.append(f'    {stage_name} -> {op_name} [dir=none]')

    r.extend(generate_ops())

    r.append('}')
    return r


def generate_outputs_gv():
    r = []
    r.append('digraph TacMan {')
    curdate = datetime.today().strftime('%Y-%m-%d')
    r.append('    label="Operations and their output nodes (red should be empty, green should be non-empty)\\n(rendered {})";'.format(curdate))
    r.append('    compound=true;')
    r.append('    style=bold')
    r.append('    fontsize=16')
    r.append('    labelloc=t')
    r.append('    rankdir=LR;')
    r.append('    size="8";')

    r.append('    node  [style="bold,rounded,filled", shape=box, fixedsize=true, width=1.6, fontname="Arial"];')

    dir = 1
    for op_name in config['operations']:
        op = config['operations'][op_name]
        if 'outputs' not in op:
            continue

        for out_name, out in iteritems(op['outputs']):
            if 'need_empty' in out:
                if out['severity'] == 'High':
                    color = '#ff7777'
                else:
                    color = 'orange'
            elif 'need_nonempty' in out:
                color = '#77ff77'
            else:
                color = 'lightgray'
            width = make_width_low(len(out_name))
            url=f'https://cs.yandex-team.ru/#!%22{out_name}%22%3A%20%7B,partner_share%2F.*%2Fconfig.json,m,arcadia,,5000'
            r.append(f'    {out_name} [tooltip="{out["name"]}", href="{url}", width={width}, fillcolor="{color}"]')
            if dir == 1:
                r.append(f'    {op_name} -> {out_name}')
            else:
                r.append(f'    {out_name} -> {op_name} [dir=back]')
        dir = -dir

    r.extend(generate_ops_block())

    r.append('}')
    return r


def generate_inputs_gv():
    r = []
    r.append('digraph TacMan {')
    curdate = datetime.today().strftime('%Y-%m-%d')
    r.append('    label="Operations and their input tables\\n(rendered {})";'.format(curdate))
    r.append('    compound=true;')
    r.append('    style=bold')
    r.append('    fontsize=16')
    r.append('    labelloc=t')
    r.append('    rankdir=LR;')
    r.append('    size="12";')

    r.append('    node  [style="bold,rounded,filled", shape=box, fixedsize=true, width=1.6, fontname="Arial"];')

    dir = 1
    for op_name in config['operations']:
        op = config['operations'][op_name]
        if 'inputs' not in op:
            continue

        for key, inp in iteritems(op['inputs']):
            inp_name = f'{str(inp)}'
            color = '#77ff77'
            width = make_width_low(len(inp_name))
            url=f'https://cs.yandex-team.ru/#!%22{key}%22%3A%20%22,partner_share%2F.*%2Fconfig.json,m,arcadia,,5000'
            r.append(f'    "{inp_name}" [title="{key}", href="{url}", width={width}, fillcolor="{color}"]')
            if dir == 1:
                r.append(f'    "{inp_name}" -> "{op_name}"')
            else:
                r.append(f'    "{op_name}" -> "{inp_name}" [dir=back]')
        dir = -dir

    r.extend(generate_ops())

    r.append('}')
    return r


def write_lines(lines, path):
    with open(path, "w") as outfile:
        outfile.write("\n".join(lines))


def run(command, shell=True):
    try:
        res = subprocess.check_output(
            command,
            shell=shell,
        ).decode("utf-8")
    except subprocess.CalledProcessError as e:
        return e.output.decode("utf-8")
    return res


def render_svg(name, lines):
    write_lines(lines, f'gv/{name}.gv')
    run(f'dot -Tsvg -ogv/{name}.svg gv/{name}.gv')
    run(f'ya upload --mds gv/{name}.svg')


def main():
    print('=====================================')
    print('REQUIRES: sudo apt install graphviz')
    print('=====================================')

    render_svg('stages', generate_stages_gv())
    render_svg('stages_tasks', generate_stages_tasks_gv())
    render_svg('operations', generate_operations_gv())
    render_svg('outputs', generate_outputs_gv())
    render_svg('inputs', generate_inputs_gv())

    print('=====================================')
    print('SVG files were generated in gv folder')
    print('Please upload generated SVG files to jing and replace links on wiki page:')
    print('https://wiki.yandex-team.ru/bannernajakrutilka/partnershare/tacman/develop')
    print('=====================================')

if __name__ == "__main__":
    main()
