#!/usr/bin/env python
# -*- coding: utf8 -*-

import logging
from datetime import date

import numpy as np

from sandbox import sdk2
from sandbox.common.errors import SandboxException
from sandbox.common.types.notification import Transport


logger = logging.getLogger(__name__)
WIKI_LINK = 'https://wiki.yandex-team.ru/bannernajakrutilka/server/preprodqueue/'


def work_days_count(start_date):
    now_date = date.today()
    return np.busday_count(start_date, now_date)


class YabsServerExperimentQueueMonitor(sdk2.Task):

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        wiki_token_vault_name = sdk2.parameters.String('Wiki token vault name', default='WIKI_TOKEN')
        number_of_days_to_experiment = sdk2.parameters.Integer('Number of working days to experiment', default=2)
        number_of_days_to_independent_experiment = sdk2.parameters.Integer('Number of working days to independent experiment', default=5)
        test_recipients = sdk2.parameters.JSON('Mail recipients for testing', default=[])

    def notify(self, username, message):
        logger.info("Send message to %s: %s", username, message)
        try:
            self.server.notification(
                subject="Experiment queue",
                body=message,
                recipients=["fdrstrok", username] if not self.Parameters.test_recipients else self.Parameters.test_recipients,
                transport=Transport.EMAIL,
                urgent=True  # send mail from sandbox-urgent@ instead of sandbox-noreply@
            )
        except Exception as e:
            raise SandboxException("Can't send notification to e-mail: %s", e.message)

    def notify_next_waiting_user(self, first_waiting_tasks, tasks_in_process=False, notify_everyone=False):
        if tasks_in_process:
            self.notify(first_waiting_tasks[0].username, "You are the first in queue for the experiment with task %s. Be ready to start tomorrow" % first_waiting_tasks[0].ticket)
            if len(first_waiting_tasks) == 2:
                self.notify(first_waiting_tasks[1].username, "You are the second in queue for the experiment with task %s. Be ready" % first_waiting_tasks[1].ticket)
        elif not notify_everyone:
            self.notify(first_waiting_tasks[0].username, "It is your turn to experiment. Please, enable your task %s" % first_waiting_tasks[0].ticket)
        else:
            for item in first_waiting_tasks:
                self.notify(item.username, "It is your turn to experiment. Please, enable your task %s" % item.ticket)

    def check_in_process(self, tasks_in_process, number_of_days):
        any_process_ends = False
        for task in tasks_in_process:
            if not task.date:
                self.notify(task.username, "Please, fill in date field in the wiki table for task %s or finish your experiment: %s" % (task.ticket, WIKI_LINK))
                any_process_ends = True
                continue
            try:
                work_days = work_days_count(task.date)
            except Exception:
                raise SandboxException("Wrong format of date for task %s", task)
            # last day of experiment
            if (number_of_days - 1) <= work_days < number_of_days:
                self.notify(task.username, "Last day of experiment %s. Please, finish it today" % task.ticket)
            # time of experiment is over
            elif work_days >= number_of_days:
                self.notify(task.username, "Your time of experiment is over, please, finish your task %s" % task.ticket)
            else:
                continue
            if not task.independent:
                any_process_ends = True
        return any_process_ends

    def on_execute(self):
        # from yabs.qa.preprod_gatekeeper.lib import get_tasks_in_process, get_waiting_tasks
        from preprod_gatekeeper import get_tasks_in_process, get_waiting_tasks
        wiki_token = sdk2.Vault.data(self.owner, self.Parameters.wiki_token_vault_name)

        tasks_in_process = get_tasks_in_process(oauth_token=wiki_token)
        tasks_in_process_affecting_statistics = get_tasks_in_process(oauth_token=wiki_token, exclude_independent=True)
        tasks_in_process_independent = filter(lambda task: task.independent, tasks_in_process)

        waiting_tasks = get_waiting_tasks(oauth_token=wiki_token)
        waiting_tasks_independent = filter(lambda task: task.independent, waiting_tasks)
        waiting_tasks_affecting_statistics = filter(lambda task: not task.independent, waiting_tasks)

        logger.info("Tasks in process: %s", tasks_in_process)
        logger.info("Waiting tasks: %s", waiting_tasks)

        any_process_ends = self.check_in_process(tasks_in_process_affecting_statistics, self.Parameters.number_of_days_to_experiment) if tasks_in_process_affecting_statistics else True

        if waiting_tasks_affecting_statistics and any_process_ends:
            self.notify_next_waiting_user(waiting_tasks_affecting_statistics[:2], tasks_in_process=bool(tasks_in_process_affecting_statistics))

        self.check_in_process(tasks_in_process_independent, self.Parameters.number_of_days_to_independent_experiment)
        count_tasks_in_process_independent = len(tasks_in_process) - len(tasks_in_process_affecting_statistics)
        logger.info("Tasks in process independent %s", count_tasks_in_process_independent)
        logger.info("Waiting tasks independent %s", waiting_tasks_independent)
        if count_tasks_in_process_independent < 5 and waiting_tasks_independent:
            free_places = 5 - count_tasks_in_process_independent
            self.notify_next_waiting_user(waiting_tasks_independent[:free_places], tasks_in_process=False, notify_everyone=True)
