import logging
import os
import pathlib2
import subprocess
import shutil

from sandbox.projects.yabs.qa import adapter_base
from sandbox import sdk2


logger = logging.getLogger(__name__)


class TaskMock(object):

    _instance = None

    def __new__(cls, *args, **kwargs):
        if not isinstance(cls._instance, cls):
            cls._instance = object.__new__(cls, *args, **kwargs)
            cls._instance._initialized = False
        return cls._instance

    def __init__(self, log_path='logs'):
        if self._initialized:
            return
        self._log_path = log_path
        if os.path.isdir(log_path):
            shutil.rmtree(log_path)
        os.makedirs(log_path)
        self._initialized = True

    def log_path(self, path):
        return pathlib2.Path(os.path.join(self._log_path, path))

    def set_info(self, message):
        logging.getLogger('task_info').info(message)


class StandaloneAdapterBase(adapter_base.ModuleAdapterBase):
    """Adapter used to run tests out of sandbox environment."""
    def __init__(self, parameters, logs_dir='logs', resource_cache_dir='resource_cache'):
        super(StandaloneAdapterBase, self).__init__(parameters, TaskMock(logs_dir))
        self._logs_dir = logs_dir
        self._resource_cache_dir = resource_cache_dir
        if not os.path.isdir(resource_cache_dir):
            os.makedirs(resource_cache_dir)

    def sync_resource(self, resource_id):
        resource_id = int(resource_id)

        cached_res_dir = os.path.join(self._resource_cache_dir, str(resource_id))
        if os.path.isdir(cached_res_dir):
            list_dir = os.listdir(cached_res_dir)
            if len(list_dir) == 1:
                logger.debug("Use resource %s from cache: %s", resource_id, os.path.abspath(cached_res_dir))
                return os.path.join(cached_res_dir, list_dir[0])

            logger.error('Faulty resource %s at local cache', resource_id)
            shutil.rmtree(cached_res_dir)

        os.makedirs(cached_res_dir)

        sbr_logger = logging.getLogger('sbr:{}'.format(resource_id))
        logger.debug('Sync sandbox resource %d', resource_id)
        with sdk2.helpers.ProcessLog(self.task_instance, logger=sbr_logger) as process_log_context:
            process_log_context.propagate = True
            cmd = [
                "sky", "get",
                "-d", cached_res_dir,
                "sbr:{}".format(resource_id)
            ]
            logger.debug("Run \"%s\"", " ".join(cmd))
            subprocess.check_call(
                cmd,
                stdout=process_log_context.stdout,
                stderr=subprocess.STDOUT
            )

        list_dir = os.listdir(cached_res_dir)
        if len(list_dir) > 1:
            shutil.rmtree(cached_res_dir)
            raise RuntimeError('Downloading resource resulted in more than one filename, what the hell. Resource: {}'.format(resource_id))

        return os.path.join(cached_res_dir, list_dir[0])

    def get_resource_attributes(self, resource_id):
        return self._client.resource[resource_id].read()['attributes']

    def get_resource_path(self, resource_id):
        return self._client.resource[resource_id].read()['file_name']
