import hashlib
import gzip
import logging
import os
import re
import shutil
import subprocess
import uuid

from enum import Enum

import sandbox.projects.common.yabs.server.requestlog as requestlog
from sandbox.projects.yabs.qa.module_base import ModuleBase
from sandbox.projects.yabs.sandbox_task_tracing import trace_calls


logger = logging.getLogger(__name__)


class EAmmoFileType(Enum):
    plain = 1
    gzip = 2


class AmmoRequestlogModule(ModuleBase):
    def __init__(self, adapter, need_initial_load=True, update_ammo=True, content_type='primary_ammo'):
        ModuleBase.__init__(self, adapter)
        if need_initial_load:
            self.reload_ammo(update_ammo=update_ammo, content_type=content_type)

    def reload_ammo(self, ammo_path=None, ammo_type=None, update_ammo=True, out_dir=os.curdir, content_type='primary_ammo'):
        logger.info('set content type to %s', content_type)
        self.adapter.set_content_type(content_type)

        if ammo_path and not ammo_type:
            ammo_type = EAmmoFileType.gzip if ammo_path.endswith('.gz') else EAmmoFileType.plain
        if not ammo_path:
            ammo_path = self.adapter.get_requestlog_path()
        if not ammo_type:
            ammo_type = self.adapter.get_requestlog_file_type()
        logging.info('Reloading ammo, path {path}, filetype {filetype}'.format(path=ammo_path, filetype=ammo_type))
        if update_ammo:
            self._ammo_path = self.generate_modified_ammo(ammo_path, ammo_type, out_dir)
        else:
            if ammo_type == EAmmoFileType.gzip:
                ungzipped_ammo_path = str(uuid.uuid4()) + '.ammo'
                with gzip.open(ammo_path, 'rb') as f_in, open(ungzipped_ammo_path, 'wb') as f_out:
                    shutil.copyfileobj(f_in, f_out)
                self._ammo_path = ungzipped_ammo_path
            elif ammo_type == EAmmoFileType.plain:
                self._ammo_path = ammo_path
        self._dplan_path = self.generate_dplan_from_ammo(self._ammo_path, out_dir)

    @trace_calls
    def update_ammo(self, bad_requests_path, content_type='primary_ammo'):
        bad_request_ids_set = set()
        with open(bad_requests_path, 'r') as f_in:
            for line in f_in:
                bad_request_ids_set.add(line.strip().lstrip('0'))
        if not bad_request_ids_set:
            return False
        request_id_header_pattern = re.compile(r'x-yabs-request-id: (\d+)', re.IGNORECASE)
        filtered_ammo_path = str(uuid.uuid4()) + '.ammo'
        with open(self.get_ammo_path(), 'r') as ammo_file, open(filtered_ammo_path, 'w') as filtered_ammo_file:
            for request, _ in requestlog.iterate(ammo_file, header_separator='\t| '):
                if not bad_request_ids_set:
                    break
                match = request_id_header_pattern.search(request)
                if not match:
                    logging.warning('No request id found in request')
                    continue
                request_id = match.group(1)
                if request_id in bad_request_ids_set:
                    requestlog.write_stpd(filtered_ammo_file, request, '<timestamp>', '<tag>')
                    bad_request_ids_set.remove(request_id)

        self.reload_ammo(filtered_ammo_path, EAmmoFileType.plain, update_ammo=False, content_type=content_type)
        return True

    def generate_modified_ammo(self, ammo_path, ammo_type, out_dir=os.curdir):
        m = hashlib.md5()
        m.update(os.path.abspath(ammo_path))
        modified_ammo_filename = m.hexdigest() + ".ammo"
        if not os.path.exists(out_dir):
            os.makedirs(out_dir)
        modified_ammo_path = os.path.abspath(os.path.join(out_dir, modified_ammo_filename))
        if os.path.isfile(modified_ammo_path):
            logging.debug("Use existing ammo: %s", modified_ammo_path)
            return modified_ammo_path

        logging.info('Generating modified ammo from %s, output to file %s', os.path.abspath(ammo_path), modified_ammo_path)
        total = 0
        parsing_failed = 0
        open_functor = gzip.open if ammo_type == EAmmoFileType.gzip else open
        with open_functor(ammo_path, 'rb') as f_in, open(modified_ammo_path, 'wb') as f_out, self.adapter.get_update_ammo_context() as update_ammo_context:
            for request, hdr_tail in requestlog.iterate(f_in, header_separator=r'\s*', single_hdr_tail=True):
                total += 1
                try:
                    out_request = update_ammo_context.update_request(request, hdr_tail)
                    if out_request:
                        requestlog.write_stpd(f_out, out_request, "<timestamp>", "<tag>")
                except Exception:
                    logging.error('Got exception while processing request', exc_info=True)
                    logging.debug('Malformed request:\n----\n%s\n----', request)
                    parsing_failed += 1

        if (float(parsing_failed) / total > 0.0002):
            raise ValueError('Too many malformed requests: {} out of {}'.format(parsing_failed, total))

        return modified_ammo_path

    def generate_dplan_from_ammo(self, ammo_path, out_dir=os.curdir):
        plan_filename = os.path.splitext(os.path.basename(ammo_path))[0] + ".plan"
        plan_path = os.path.abspath(os.path.join(out_dir, plan_filename))
        if os.path.isfile(plan_path):
            logging.debug("Use existing plan: %s", plan_path)
            return plan_path

        dplanner_executable_path = self._adapter.get_dplanner_executable_path()
        logging.info('Generating dplan from ammo file %s, output to file %s', ammo_path, plan_path)
        commandline = [
            dplanner_executable_path,
            '-l', os.path.abspath(ammo_path),
            '-o', plan_path,
            '-t', 'phantom',
            '-h', 'localhost'
        ]
        logging.debug("Run \"%s\"", " ".join(commandline))
        subprocess.check_call(commandline)
        return plan_path

    def get_ammo_path(self):
        return self._ammo_path

    def get_dplan_path(self):
        return self._dplan_path
