
import random
import logging
import itertools

from collections import defaultdict

from sandbox import sdk2
from sandbox.common.errors import TaskError, TaskFailure
from sandbox.common.types.task import ReleaseStatus

from sandbox.projects.common.dolbilka.resources import DPLANNER_EXECUTABLE
from sandbox.projects.common.yabs.server.util.libs.request_parser import HTTPRequest
from sandbox.projects.yabs.qa.adapter_base.sandbox import SandboxAdapterBase
from sandbox.projects.yabs.qa.ammo_module.requestlog.adapters.interface import AmmoRequestlogModuleAdapterBase
from sandbox.projects.yabs.qa.ammo_module.requestlog.adapters.yabs_stat.sandbox.parameters import AmmoRequestlogYabsStatParameters
from sandbox.projects.yabs.qa.ammo_module.requestlog import EAmmoFileType, AmmoRequestlogModule
from sandbox.projects.yabs.qa.utils import general as general_qa_utils


class AmmoRequestlogModuleYabsStatSandboxAdapter(AmmoRequestlogModuleAdapterBase, SandboxAdapterBase):
    def __init__(self, parameters, task_instance, requestlog_path):
        SandboxAdapterBase.__init__(self, parameters, task_instance)
        self._http_parser = HTTPRequest()
        self._init_requestlog_path = requestlog_path

    @staticmethod
    def get_init_parameters_class():
        return AmmoRequestlogYabsStatParameters

    def create_module(self):
        return AmmoRequestlogModule(self)

    def get_requestlog_path(self):
        return self._init_requestlog_path

    def get_requestlog_file_type(self):
        return EAmmoFileType.plain

    def update_ammo_initialize(self):
        self._written_request_codes = {}
        self._unwritten_request_codes = {}
        self._total_good_size = 0

    def update_request_method(self, request, hdr_tail):
        self._http_parser.from_string(request)
        if self._http_parser.handler == 'ping':
            return None
        if random.random() > self.parameters.content_fraction and self._http_parser.handler in ('content', 'write_wide', 'get_data'):
            return None
        request_key = (self._http_parser.headers.get('x-yabs-request-id'), self._http_parser.handler)

        code = hdr_tail.split('\t')[2]
        if general_qa_utils.is_bad_http_code(code):
            if request_key not in self._written_request_codes:
                self._unwritten_request_codes[request_key] = code
        else:
            if request_key in self._written_request_codes:
                logging.warning("Skipping duplicate write of request with key %s", request_key)
                return None
            self._written_request_codes[request_key] = code
            self._unwritten_request_codes.pop(request_key, None)
            self._total_good_size += len(request)
            return request

    def update_ammo_finalize(self):
        requests_by_code = defaultdict(int)
        for code in itertools.chain(self._unwritten_request_codes.itervalues(), self._written_request_codes.itervalues()):
            requests_by_code[code] += 1

        logging.info(
            "Request codes in yabs-stat log:\n" +
            '\n'.join('{}: {}'.format(code, requests_by_code[code]) for code in sorted(requests_by_code))
        )
        total_count = len(self._written_request_codes) + len(self._unwritten_request_codes)

        client_error_count = sum(count for code, count in requests_by_code.iteritems() if general_qa_utils.is_http_client_error(code))

        if client_error_count >= (self.parameters.client_threshold * total_count):
            raise TaskFailure(
                "Cannot shoot: too many 4xx codes in generated yabs-stat log ({} out of {})".format(
                    client_error_count, total_count
                )
            )

        if len(self._unwritten_request_codes) >= (self.parameters.bad_rate_threshold * total_count):
            raise TaskError(
                "Cannot shoot: too many bad requests in generated yabs-stat log ({} out of {})".format(
                    len(self._unwritten_request_codes), total_count
                )
            )
        logging.info("Average request size for shooting yabs-stat: {} bytes".format(self._total_good_size / total_count))

    def get_dplanner_executable_path(self):
        return self.sync_resource(sdk2.Resource.find(DPLANNER_EXECUTABLE, attrs={'released': ReleaseStatus.STABLE}).order(-sdk2.Resource.id).first())
