import re
import logging

from jinja2 import Environment


from datetime import timedelta
from yt.wrapper import ypath_join
from yt.wrapper.errors import YtCypressTransactionLockConflict
from sandbox.projects.common.decorators import retries
from sandbox.projects.yabs.qa.bases.sample_tables.sampling import sample_tables, merge_sampled_tables, alter_merged_tables
from sandbox.projects.yabs.qa.bases.sample_tables.tables import (
    create_sampling_tables_config,
    check_sampled_intermediate_tables_existence,
    check_merged_intermediate_tables_existence,
    link_resulting_tables,
)
from sandbox.projects.yabs.qa.bases.sample_tables.closest_event_logs import choose_closest_event_logs


logger = logging.getLogger(__name__)
TRANSACTION_TITLE_TEMPLATE = 'sandbox_task_id:{}'


def get_task_id_from_transaction(yt_client, tx_info):
    try:
        return int(re.match(TRANSACTION_TITLE_TEMPLATE.format('(?P<task_id>[0-9]+)'), tx_info['title']).groupdict()['task_id'])
    except (KeyError, ValueError, AttributeError):
        logger.debug('Got exception while trying to parse transaction_title from %s', tx_info, exc_info=True)
    return int(yt_client.get(ypath_join('//sys/transactions', tx_info['id'], '@task_id')))


@retries(max_tries=3, delay=5)
def try_lock(path, tx, yt_client):
    try:
        yt_client.lock(path, mode='exclusive')
        return True, None
    except YtCypressTransactionLockConflict as exc:
        logger.debug('Exception object error: %s', exc.error)
        logger.debug('Exception object args: %s', exc.args)
        logger.debug('Exception object attributes: %s', exc.attributes)
        winner_transaction = exc.error['attributes']['winner_transaction']
        winning_task_id = get_task_id_from_transaction(yt_client, winner_transaction)
        logger.info('Winning task_id: %s', winning_task_id)

    return False, winning_task_id


def run_sample_tables(
    input_archive_path,
    shallow_copy_path,
    sampled_tables_path,
    query_template_text,
    tables,
    yt_client,
    yt_client_lock,
    yt_client_eventlog,
    yt_pool,
    yql_client,
    yt_cluster='hahn',
    task_id=None,
    dry_run=True,
    need_mount_tables={},
):
    table_configs = create_sampling_tables_config(shallow_copy_path, sampled_tables_path, yt_client, tables=tables, need_mount_tables=need_mount_tables)

    # Use separate path for lock, because alter_table cannot execute in transaction
    sampling_lock_path = ypath_join(sampled_tables_path, 'sampling_lock')

    logger.info('Trying to create output prefix and lock node: %s', sampling_lock_path)
    yt_client_lock.create('map_node', sampling_lock_path, recursive=True, ignore_existing=True)

    did_sampling = False

    with yt_client_lock.Transaction(attributes={'task_id': str(task_id), 'title': TRANSACTION_TITLE_TEMPLATE.format(task_id)}, ping=True) as tx:
        is_success, winning_task_id = try_lock(sampling_lock_path, tx, yt_client_lock)
        if not is_success:
            logger.info('Cannot take lock at %s because %s took it first', sampled_tables_path, winning_task_id)
            return is_success, did_sampling, winning_task_id, table_configs

        logger.info("Successfully got lock at %s, will try to sample tables", sampled_tables_path)

        missing_intermediate_table_configs = check_sampled_intermediate_tables_existence(table_configs, yt_client)
        if missing_intermediate_table_configs:
            logger.info("Some intermediate tables do not exist, will launch sampling YQL")
            event_logs = choose_closest_event_logs(
                yt_client_eventlog,
                input_archive_path,
                '//logs/bs-event-log',
                timedelta(days=1)
            )
            query_template = Environment().from_string(query_template_text)
            query = query_template.render(
                yt_pool=yt_pool,
                cluster_name=yt_cluster,
                sampling_table_configs=missing_intermediate_table_configs,
                event_logs=event_logs,
                task_id=task_id,
            )
            logger.debug("Rendered query:\n%s", query)
            did_sampling = True
            if not dry_run:
                logger.debug('Setting atribute: %s', sampled_tables_path + '/@latest_sampling_task_id')
                yt_client.set(sampled_tables_path + '/@latest_sampling_task_id', task_id)
                sample_tables(missing_intermediate_table_configs, query, yql_client, yt_client, title='Sampling input-archive by task_id {} via YQL'.format(task_id))
        else:
            logger.info("All intermediate tables exist")

        missing_merged_table_configs = check_merged_intermediate_tables_existence(table_configs, yt_client)
        if missing_merged_table_configs:
            logger.info("Some merged tables do not exist, will start copying")
            if not dry_run:
                merge_sampled_tables(missing_merged_table_configs, yt_client)
        else:
            logger.info("All merged tables exist")

        alter_merged_tables(table_configs, yt_client)
        link_resulting_tables(table_configs, yt_client)

    return is_success, did_sampling, winning_task_id, table_configs
