import logging
import json
from yt.wrapper import OperationsTracker
from sandbox.projects.yabs.qa.utils.yt_utils import mount_tables_batched, freeze_tables_batched
from sandbox.projects.yabs.qa.bases.sample_tables.tables import create_dynamic_table_empty_copy


logger = logging.getLogger(__name__)


def get_yt_user_attributes(yt_client, path):
    logger.debug('Getting attributes: %s', path + '/@user_attributes')
    return json.loads(yt_client.get(path + '/@user_attributes', format="json"))


def set_yt_attributes(yt_client, path, attrs):
    for attr in attrs:
        logger.debug('Setting atribute: %s', path + '/@' + attr)
        yt_client.set(path + '/@' + attr, attrs[attr])


def sample_tables(table_configs, yql_sampling_query, yql_client, yt_client, title='Sampling input-archive via YQL'):
    with yt_client.Transaction(ping=True) as tx:
        logger.debug('Sampling YQL query: %s', yql_sampling_query)
        request = yql_client.query(yql_sampling_query, syntax_version=1, title=title)
        result = request.run(transaction_id=tx.transaction_id)
        logger.debug('Sampling YQL operation: %s', request.share_url)
        logger.debug('Sampling YQL operation result: %s', result)

        for table_config in table_configs:
            attrs = get_yt_user_attributes(
                yt_client,
                table_config.absolute_input_path
            )
            set_yt_attributes(
                yt_client,
                table_config.intermediate_absolute_output_path,
                {attr: attrs[attr] for attr in attrs if attr.startswith('_yql_proto_field_')}
            )


def merge_sampled_tables(table_configs, yt_client):
    with yt_client.Transaction() as tx:
        logging.debug('Copying tables in transaction %s', tx.transaction_id)
        tables_to_merge = [table_config for table_config in table_configs if table_config.is_dynamic]
        tables_to_link = [table_config for table_config in table_configs if not table_config.is_dynamic]

        for table_config in tables_to_link:
            logger.debug('Linking %s to %s', table_config.intermediate_absolute_output_path, table_config.intermediate_merged_absolute_output_path)
            yt_client.link(table_config.intermediate_absolute_output_path, table_config.intermediate_merged_absolute_output_path, force=True)

        if tables_to_merge:
            tracker = OperationsTracker()
            for table_config in tables_to_merge:
                if yt_client.exists(table_config.intermediate_merged_absolute_output_path):
                    logger.debug(
                        'Removing %s %s',
                        yt_client.get(table_config.intermediate_merged_absolute_output_path.rstrip('&') + '&/@type'),
                        table_config.intermediate_merged_absolute_output_path,
                    )
                    yt_client.remove(table_config.intermediate_merged_absolute_output_path.rstrip('&') + '&')

                logger.debug('Merging %s to %s', table_config.intermediate_absolute_output_path, table_config.intermediate_merged_absolute_output_path)
                create_dynamic_table_empty_copy(table_config.absolute_input_path, table_config.intermediate_merged_absolute_output_path, yt_client)
                tracker.add(
                    yt_client.run_merge(
                        table_config.intermediate_absolute_output_path,
                        table_config.intermediate_merged_absolute_output_path,
                        spec={'input_query': '*'},
                        mode='ordered'
                    )
                )
            tracker.wait_all()


def alter_merged_tables(table_configs, yt_client):
    tables_to_merge = [table_config for table_config in table_configs if table_config.is_dynamic]
    if tables_to_merge:
        for table_config in tables_to_merge:
            logger.debug('Altering dynamic table: %s', table_config.absolute_output_path)
            yt_client.alter_table(table_config.intermediate_merged_absolute_output_path, dynamic=True)

    unmounted_tables = []
    for table_config in table_configs:
        if table_config.is_dynamic and table_config.need_mount and str(yt_client.get_attribute(table_config.intermediate_merged_absolute_output_path, 'tablet_state')) == 'unmounted':
            unmounted_tables.append(table_config.intermediate_merged_absolute_output_path)

    logger.info('Mount tables: %s', unmounted_tables)
    mount_tables_batched(unmounted_tables, yt_client)

    logger.info('Freeze tables: %s', unmounted_tables)
    freeze_tables_batched(unmounted_tables, yt_client)
