# -*- coding: utf-8 -*-

from __future__ import print_function, absolute_import, division

import logging
import textwrap
import six
import re

from sandbox.projects.yabs.qa.utils import yt_utils

from sandbox.projects.yabs.sandbox_task_tracing import trace_calls
from sandbox.projects.yabs.sandbox_task_tracing.wrappers.sandbox.projects.common.yql import run_query


logger = logging.getLogger(__name__)

RESHARD_ERROR_PATTERN = 'Cannot reshard table since tablet .* is not unmounted'


def get_key_and_table_paths(tables):
    from yt.wrapper import ypath_join
    key_paths = {}
    table_paths = set()
    need_mount = set()
    for desc in tables:
        key = desc.get('id', '')
        path = desc.get('path', '')
        table = desc.get('table', '')

        if key in key_paths and key_paths[key] != path:
            raise Exception('Inconsistent paths for path key "%s": "%s" and "%s"' % (key, key_paths[key], path))

        key_paths[key] = path
        table_paths.add(ypath_join(path, table) if table else path)

        if desc.get('need_mount', False):
            need_mount.add(ypath_join(path, table) if table else path)

    return key_paths, table_paths, need_mount


def get_tables_state_map(yt, tables):
    return {
        str(table): yt.get_attribute(table, 'tablet_state')
        for table in tables
    }


def get_tables_destination(destination):
    from yt.wrapper import ypath_join
    return ypath_join(destination, 'yt_tables')


def destination_table_path(source_table_path, archive_root):
    tables_destination = get_tables_destination(archive_root)
    return _get_destination_path(tables_destination, source_table_path)


@trace_calls(save_arguments=(1, 'tables'))
def save_input(
    yt,
    tables,
    destination,
    dynamic_bundle_name=None,
    no_quota_attrs=True,
    recreate_links=True,
    tables_filtred_by_orderid=None,
    baseno_list=None,
    baseno_by_orederid_table=None,
    yt_proxy='hahn',
    yql_token=None,
    yt_pool=None
):
    """
    tables - list of dicts: [
        {
            'id': 'table1',
            'path': '//home/Table1'
        },
        {
            'id': 'table2',
            'path': '//home',
            'table': 'Table2',
            'need_mount': False
        }
    ]
    """
    from yt.wrapper import ypath_split, YtHttpResponseError
    key_paths = {}
    table_paths = set()
    tables_filtred_by_orderid = tables_filtred_by_orderid or []
    key_paths, table_paths, need_mount = get_key_and_table_paths(tables)

    with yt.Transaction():
        destination_prefix, child_key = ypath_split(destination)
        logger.info("Acquiring shared lock on %s with child_key %s", destination_prefix, child_key)
        yt.lock(destination_prefix, mode='shared', child_key=child_key, waitable=True, wait_for=10 * 3600 * 1000)

        yt.create('map_node', destination, ignore_existing=True)

        logger.info("Acquiring exclusive lock on %s", destination)
        yt.lock(destination, mode='exclusive', waitable=True, wait_for=10 * 3600 * 1000)
        logger.info("Locked")

        _check_tables_existence(yt, table_paths)

        tables_destination = get_tables_destination(destination)
        _save_new_tables(
            yt,
            table_paths,
            tables_destination,
            no_quota_attrs,
            tables_filtred_by_orderid,
            recreate_links=recreate_links,
        )

    for src_path in tables_filtred_by_orderid:
        if src_path in table_paths:
            dst_path = _get_destination_path(tables_destination, src_path)
            if yt.exists(dst_path):
                logger.info('Skip copying table "%s" – node "%s" already exists', src_path, dst_path)
                continue

            # Select and insert only necessary orders
            _copy_data_by_order_id(src_path, dst_path, baseno_list, baseno_by_orederid_table, yt_proxy, yql_token, yt_pool)

    src_dynamic_tables = list(
        filter(
            lambda path_: yt.get_attribute(path_, 'dynamic', default=False) and path_ in need_mount,
            table_paths
        )
    )

    dst_dynamic_tables = list(
        _get_destination_path(tables_destination, path_)
        for path_ in src_dynamic_tables
    )

    reshard_tables = {
        _get_destination_path(tables_destination, path_): yt.get_attribute(path_, 'tablet_count', 1)
        for path_ in need_mount
    }

    logger.info('Converting tables to dynamic: %s', map(str, dst_dynamic_tables))
    convert_tables_to_dynamic(dst_dynamic_tables, yt)

    dst_dynamic_tables_state_map = get_tables_state_map(yt, dst_dynamic_tables)

    for table_ in reshard_tables:
        if table_ in dst_dynamic_tables and dst_dynamic_tables_state_map[table_] == 'unmounted':
            if dynamic_bundle_name is not None:
                yt.set_attribute(table_, 'tablet_cell_bundle', dynamic_bundle_name)
            try:
                yt.reshard_table(table_, tablet_count=reshard_tables[table_])
            except YtHttpResponseError as error:
                if error.find_matching_error(predicate=lambda e: re.match(RESHARD_ERROR_PATTERN, e.message) is not None) is None:
                    raise error

    with yt.Transaction():
        logger.info("Acquiring exclusive lock on %s", tables_destination)
        yt.lock(tables_destination, mode='exclusive', waitable=True, wait_for=10 * 3600 * 1000)
        logger.info("Locked")

        unmounted_tables = list(filter(
            lambda table_: dst_dynamic_tables_state_map[table_] == 'unmounted',
            dst_dynamic_tables_state_map
        ))
        logger.info('Unmounted tables: %s', unmounted_tables)
        mounted_tables = list(filter(
            lambda table_: dst_dynamic_tables_state_map[table_] == 'mounted',
            dst_dynamic_tables_state_map
        ))
        logger.info('Mounted tables: %s', mounted_tables)

        logger.info('Mount tables: %s', unmounted_tables)
        yt_utils.mount_tables_batched(unmounted_tables, yt)
        mounted_tables += unmounted_tables

        logger.info('Freeze tables: %s', mounted_tables)
        yt_utils.freeze_tables_batched(mounted_tables, yt)

    return _build_table_key_spec(key_paths, tables_destination)


def _check_tables_existence(yt, paths):
    for path in paths:
        if not yt.exists(path):
            raise Exception('Input table "%s" doesn\'t exist' % path)


def _save_new_tables(
    yt,
    paths,
    destination,
    save_input_no_quota_attrs,
    tables_filtred_by_orderid,
    recreate_links=False,
):
    for path in paths:
        if path in tables_filtred_by_orderid:
            continue
        save_table_if_new(
            yt,
            src_path=path,
            dst_path=_get_destination_path(destination, path),
            save_input_no_quota_attrs=save_input_no_quota_attrs,
            destination_to_recreate_links=destination if recreate_links else None,
        )


@trace_calls
def save_table_if_new(yt, src_path, dst_path, save_input_no_quota_attrs, destination_to_recreate_links=None):
    if yt.exists(dst_path):
        logger.info('Skip copying table "%s" – node "%s" already exists', src_path, dst_path)
        return

    if destination_to_recreate_links:
        target_file = yt.get_attribute(src_path + '&', 'target_path', default=None)
        if target_file:
            dst_target_file = _get_destination_path(destination_to_recreate_links, target_file)
            save_table_if_new(
                yt,
                target_file,
                dst_target_file,
                save_input_no_quota_attrs,
                destination_to_recreate_links,
            )
            logger.info('Link table "%s" to "%s"', dst_path, dst_target_file)
            yt.link(dst_target_file, dst_path, recursive=True)
            return

    logger.info('Copying table "%s" to "%s"', src_path, dst_path)
    is_default_medium = (yt.get_attribute(src_path, 'primary_medium', default='default') == 'default')
    is_default_tablet_cell_bundle = (yt.get_attribute(src_path, 'tablet_cell_bundle', default='default') == 'default')
    if yt.get_attribute(src_path, 'dynamic', default=False):
        logger.debug('Table %s is dynamic, copy table via merge', src_path)
        # Can't use yt.copy because it requires to freeze production table
        _copy_dynamic_table(yt, src_path, dst_path)
    elif save_input_no_quota_attrs and (not is_default_medium or not is_default_tablet_cell_bundle):
        # BSSERVER-9231
        yt.lock(src_path, mode='snapshot')
        logger.debug('Copy table via merge')
        _copy_table_via_merge(yt, src_path, dst_path)
    else:
        logger.debug('Copy table via yt.copy')
        yt.copy(src_path, dst_path, recursive=True)


@trace_calls
def _copy_dynamic_table(yt, src_path, dst_path):
    yt.lock(src_path, mode='snapshot')
    src_attributes = {}
    dst_attributes = {
        'optimize_for': yt.get_attribute(src_path, 'optimize_for', default='lookup'),
    }
    _copy_table_via_merge(yt, src_path, dst_path, src_attributes, dst_attributes)


@trace_calls
def _copy_table_via_merge(yt, src_path, dst_path, src_attributes=None, dst_attributes=None):
    logger.debug('src_attributes=%s; dst_attributes=%s', src_attributes, dst_attributes)
    yt.create('table', dst_path, attributes=dst_attributes, recursive=True)
    operation = yt.run_merge(
        yt.TablePath(src_path, attributes=src_attributes),
        dst_path,
        mode='ordered',
        spec={
            'weight': 10005000,  # Many Sandbox tasks might be waiting for this one merge, run it as fast as possible
            'enable_dynamic_store_read': False,  # don't use dynamic store read, it may corrupt caesar profiles
        },
        sync=False,
    )
    logger.debug('Run operation %s', operation.url)
    operation.wait()


def _build_table_key_spec(key_paths, destination):
    return {key: _get_destination_path(destination, path) for key, path in six.iteritems(key_paths)}


def _get_destination_path(destination, path):
    from yt.wrapper import ypath_join
    return ypath_join(destination, path.lstrip('/'))


def convert_tables_to_dynamic(paths, client):
    for path in paths:
        if not client.get_attribute(path, 'dynamic', default=False):
            client.alter_table(path, dynamic=True)


@trace_calls
def _copy_data_by_order_id(src_path, dst_path, baseno_list, baseno_by_orederid_table, yt_proxy, yql_token, yt_pool):
    logger.info('Save data from %s only for baseno %s', src_path, baseno_list)
    yql_query = textwrap.dedent('''
        PRAGMA yt.Pool = "{yt_pool}";
        PRAGMA AnsiInForEmptyOrNullableItemsCollections;
        PRAGMA yt.OperationSpec = "{{enable_dynamic_store_read=%false}}";

        $order_id = SELECT OrderID FROM {proxy}.`{baseno_by_orderid}` WHERE BaseNo IN ({baseno_list});

        INSERT INTO {proxy}.`{dst_path}` WITH TRUNCATE
        SELECT * FROM {proxy}.`{src_path}` WHERE OrderID IN $order_id;
    ''').format(yt_pool=yt_pool, proxy=yt_proxy, baseno_by_orderid=baseno_by_orederid_table, dst_path=dst_path, src_path=src_path, baseno_list=','.join(map(str, baseno_list)))
    run_query(
        query_id='Copy table with necessary baseno',
        query=yql_query,
        yql_token=yql_token,
        db=yt_proxy,
        wait=True,
        syntax_version=1)
