import json
import itertools
import logging

from difflib import ndiff

from sandbox.projects.yabs.sandbox_task_tracing import trace_calls


NO_DIFF_MESSAGE = 'No diff'


def sizeof_fmt(num, suffix='B', addfmt=''):
    for unit in ['', 'Ki', 'Mi', 'Gi']:
        if abs(num) < 1024.0:
            return ("%" + addfmt + "3.1f %s%s") % (num, unit, suffix)
        num /= 1024.0
    return ("%" + addfmt + ".1f %s%s") % (num, 'Gi', suffix)


@trace_calls
def compare_db_sizes(baseline_sizes, test_sizes, max_diff_percent):
    baseline_total = sum(baseline_sizes.itervalues())
    test_total = sum(test_sizes.itervalues())

    diff_percent = 100.0 * (test_total / baseline_total - 1.0)

    has_diff = abs(diff_percent) > max_diff_percent
    report = ["Total binary bases size changed by {delta} ~ {percent:+.4f}% ({baseline} -> {test})".format(
        delta=sizeof_fmt(test_total - baseline_total, addfmt='+'),
        percent=diff_percent,
        baseline=sizeof_fmt(baseline_total),
        test=sizeof_fmt(test_total),
    )]

    if has_diff and diff_percent < 0:
        report += [
            "Total binary bases size of yabs-server DECREASED.",
            "If your commit can indeed be expected to cause such an improvement,",
            "please mark the problem as RESOLVED and do not complain."
        ]

    for db in sorted(baseline_sizes.viewkeys() & test_sizes.viewkeys()):
        if baseline_sizes[db] == test_sizes[db]:
            continue
        bs = baseline_sizes[db]
        ts = test_sizes[db]
        report.append(
            "{db}: {delta} ~ {percent:+.4f}% ({baseline} -> {test})".format(
                db=db,
                delta=sizeof_fmt(ts - bs, addfmt='+'),
                percent=100.0 * (ts / bs - 1.0),
                baseline=sizeof_fmt(bs),
                test=sizeof_fmt(ts),
            )
        )

    for db in sorted(baseline_sizes.viewkeys() - test_sizes.viewkeys()):
        report.append("{} disappeared ({})".format(db, sizeof_fmt(baseline_sizes[db])))

    for db in sorted(test_sizes.viewkeys() - baseline_sizes.viewkeys()):
        report.append("{} added ({})".format(db, sizeof_fmt(test_sizes[db])))

    return has_diff, '\n'.join(report)


@trace_calls
def compare_chkdb(baseline_chkdb, test_chkdb, sync_resource):
    diffs = {}
    no_diff_dbs = []
    same_dbs = []

    for db in sorted(baseline_chkdb.viewkeys() & test_chkdb.viewkeys()):
        if not baseline_chkdb[db] or not test_chkdb[db]:
            continue
        if baseline_chkdb[db] == test_chkdb[db]:
            same_dbs.append(db)
            continue
        logging.info('Downloading chkdb resources for %s', db)
        baseline_chkdb_path = sync_resource(int(baseline_chkdb[db]))
        test_chkdb_path = sync_resource(int(test_chkdb[db]))
        try:
            with open(baseline_chkdb_path) as baseline_file, open(test_chkdb_path) as test_file:
                baseline = json.load(baseline_file)
                test = json.load(test_file)
        except Exception:
            with open(baseline_chkdb_path) as baseline_file, open(test_chkdb_path) as test_file:
                diff = ''.join(l for l in ndiff(baseline_file.readlines(), test_file.readlines()) if l.startswith(('+', '-')))
        else:
            diff = get_one_chkdb_diff(baseline, test)

        if diff:
            diffs[db] = diff
        else:
            no_diff_dbs.append(db)

    has_chkdb_diff = any(not db.startswith(('resource', 'dssm')) for db in diffs)

    def _iter_report():
        if same_dbs:
            yield "Same resources: {}".format(' '.join(sorted(same_dbs)))
            yield ""
        if no_diff_dbs:
            yield "No diff: {}".format(' '.join(sorted(no_diff_dbs)))
            yield ""
        if diffs:
            yield "Bases with diff:"
            for db in sorted(diffs):
                yield ""
                yield '{}:'.format(db)
                yield diffs[db]

    return has_chkdb_diff, '\n'.join(_iter_report())


def get_one_chkdb_diff(baseline, test):
    return '\n'.join(itertools.chain.from_iterable(_iter_one_chkdb_diff(baseline, test)))


def _iter_one_chkdb_diff(baseline, test):
    for key in sorted(baseline.viewkeys() | test.viewkeys()):
        baseline_items = baseline.get(key, [])
        test_items = test.get(key, [])
        if baseline_items == test_items:
            continue
        yield ['{}:'.format(key)]
        try:
            baseline_map = _make_submap(baseline_items)
            test_map = _make_submap(test_items)
        except NotSubmap:
            if len(baseline_items) == len(test_items):
                for baseline_item, test_item in itertools.izip(baseline_items, test_items):
                    if baseline_item != test_item:
                        yield _pair_diff(baseline_item, test_item)
            else:
                yield _lists_diff(baseline_items, test_items)
        else:
            for subkey in sorted(baseline_map.viewkeys() | test_map.viewkeys()):
                yield _pair_diff(baseline_map.get(subkey), test_map.get(subkey))


def _pair_diff(a, b):
    if a == b:
        return
    if a is not None:
        yield '    - ' + json.dumps(a, sort_keys=True)
    if b is not None:
        yield '    + ' + json.dumps(b, sort_keys=True)


def _lists_diff(a, b):
    lines = [[json.dumps(l, sort_keys=True) for l in lst] for lst in (a, b)]

    diff = list(' ' * 4 + line for line in ndiff(*lines) if line.startswith(('+', '-')))
    logging.info("_items_diff %s %s", lines, diff)
    return diff


class NotSubmap(Exception):
    pass


def _make_submap(items):
    result = {}
    for item in items:
        try:
            key = item['Key']
        except KeyError:
            raise NotSubmap("No key in {}".format(item))
        if key in result:
            raise NotSubmap("Duplicate key {}".format(key))
        result[key] = item
    return result
