import logging
import os
import subprocess
import time
import uuid

from sandbox.projects.yabs.qa.module_base import ModuleBase
from sandbox.common.errors import TaskFailure
from sandbox.sdk2.helpers import ProcessLog


class DolbiloModule(ModuleBase):
    class DolbiloSessionException(TaskFailure):
        pass

    def __init__(self, adapter):
        ModuleBase.__init__(self, adapter)
        self._d_executor_path = self.adapter.get_d_executor_path()

    def _construct_command_line(self, dplan_path, output_path, port, store_dump, shoot_threads=None, no_body_encoding=True):
        shoot_threads = shoot_threads if shoot_threads else self.adapter.get_shoot_threads()
        cmdline = [
            self._d_executor_path,
            '--plan-file', dplan_path,
            '--output', output_path,
            '--replace-host', 'localhost',
            '--replace-port', port,
            '--mode', self.adapter.get_shoot_mode(),
            '--simultaneous', shoot_threads,
            '--timeout', self.adapter.get_request_timeout(),
        ]
        logging.debug('Threads in shoot - %s', shoot_threads)
        if no_body_encoding:
            cmdline.append('--no-body-encoding')
        if self.adapter.get_shoot_mode_args():
            cmdline += self.adapter.get_shoot_mode_args()
        if self.adapter.get_shoot_request_limit():
            cmdline.extend([
                '--queries-limit', self.adapter.get_shoot_request_limit()
            ])
        if self.adapter.get_store_plan_in_memory():
            cmdline.append('--plan-memory')
        if self.adapter.get_circular_session():
            cmdline.append('--circular')
        if store_dump:
            cmdline.append('-d')
        return cmdline

    def output_dump_path(self, out_dir=''):
        if out_dir and not os.path.exists(out_dir):
            os.makedirs(out_dir)
        return os.path.join(out_dir, '{}.dump'.format(uuid.uuid4()))

    def shoot(self, sut_service, dplan_path, store_dump=True, no_body_encoding=True, out_dir=os.curdir):
        output_path = self.output_dump_path(out_dir)
        cmdline = self._construct_command_line(
            dplan_path=dplan_path,
            output_path=output_path,
            port=sut_service.get_port(),
            store_dump=store_dump,
            no_body_encoding=no_body_encoding,
        )
        logging.info("Save shoot results to %s", output_path)
        with ProcessLog(self.adapter.task_instance, 'shoot_session') as process_log:
            process_log.logger.propagate = True
            cmdline = map(str, cmdline)
            logging.debug("Run %s", " ".join(cmdline))
            subprocess.check_call(cmdline, stdout=process_log.stdout, stderr=subprocess.STDOUT)
        return output_path

    def shoot_and_watch(self, sut_service, dplan_path, store_dump=True, shoot_threads=None, watch_period=10, out_dir='.'):
        output_path = self.output_dump_path(out_dir)
        cmdline = self._construct_command_line(
            dplan_path=dplan_path,
            output_path=output_path,
            port=sut_service.get_port(),
            store_dump=store_dump,
            shoot_threads=shoot_threads,
        )
        with ProcessLog(self.adapter.task_instance, 'shoot_session') as process_log:
            process_log.logger.propagate = True
            shoot_process = subprocess.Popen(map(str, cmdline), stdout=process_log.stdout, stderr=subprocess.STDOUT)
            while True:
                if not sut_service.is_active():
                    shoot_process.kill()
                    shoot_process.communicate()
                    raise self.DolbiloSessionException('Watched service under test shutdown unexpectedly')
                shoot_return_code = shoot_process.poll()
                if shoot_return_code is not None:
                    if shoot_return_code:
                        raise self.DolbiloSessionException('Shoot session ended with code {}'.format(shoot_return_code))
                    else:
                        logging.info('Shoot session ended successfully')
                        return output_path
                time.sleep(watch_period)
