import logging

from sandbox.projects.yabs.qa.hamster.deploy import (
    DeployServiceNotFound,
    is_deploy_unit_ready,
)
from sandbox.projects.yabs.qa.hamster.nanny.utils import (
    is_service_active as is_nanny_service_active,
    NannyServiceNotFound,
)
from sandbox.projects.yabs.qa.hamster.spec import ExternalServiceType, UnknownExternalServiceType
from sandbox.projects.yabs.qa.utils.resource import json_from_resource


logger = logging.getLogger(__name__)


class HamsterNotFound(Exception):
    """Raised when hamster does not exist.
    """


def check_hamster_readiness(resource_id, nanny_token, yp_token):
    """Check if hamster is ready to accept incoming requests.

    :param resource_id: Hamster endpoint resource id
    :type resource_id: int
    :param nanny_token: Nanny token
    :type nanny_token: str
    :param yp_token: YP token, used to check Deploy services
    :type yp_token: str
    :return: Is hamster ready to accept incoming requests
    :rtype: bool
    :raises UnknownExternalServiceType: If service type is unknown
    :raises HamsterNotFound: If hamster does not exist
    """
    endpoint_data = json_from_resource(resource_id)

    service_id = endpoint_data["service_id"]
    service_type = endpoint_data["service_type"]

    if service_type == ExternalServiceType.NANNY.value:
        try:
            return is_nanny_service_active(nanny_token, service_id)
        except NannyServiceNotFound:
            raise HamsterNotFound(service_id)

    if service_type == ExternalServiceType.DEPLOY.value:
        _, deploy_unit_id = endpoint_data["endpoint_set"].rsplit(".", 1)
        try:
            return is_deploy_unit_ready(service_id, deploy_unit_id, yp_token)
        except DeployServiceNotFound:
            raise HamsterNotFound(service_id)

    raise UnknownExternalServiceType(service_type)
