import logging
import os
import subprocess


logger = logging.getLogger(__name__)


class DeployServiceNotFound(Exception):
    """Raised when deploy service does not exist.
    """


def put_stage(stage_path, yp_token):
    from sandbox import sdk2

    # get ya
    ya = sdk2.vcs.svn.Arcadia.export("arcadia:/arc/trunk/arcadia/ya", os.path.realpath("ya"))

    # run ya tool dctl put stage stage.yml
    cmd = [ya, "tool", "dctl", "put", "stage", stage_path]
    logger.debug("Run: \"%s\"", " ".join(cmd))
    env = os.environ
    env["DCTL_YP_TOKEN"] = yp_token
    with sdk2.helpers.ProcessLog(sdk2.Task.current, logger="ya_tool_dctl") as pl:
        subprocess.check_call(
            cmd,
            stdout=pl.stdout,
            stderr=pl.stderr,
            env=env,
        )


def is_deploy_unit_ready(stage_id, deploy_unit_id, yp_token):
    from yt.orm.library.common import NoSuchObjectError
    from yp.client import YpClient

    yp_client = YpClient("xdc", config={"token": yp_token})
    status_selector = os.path.join("/status", "deploy_units", deploy_unit_id, "ready", "status")
    try:
        responses = yp_client.get_object("stage", stage_id, selectors=[status_selector])
    except NoSuchObjectError:
        raise DeployServiceNotFound(stage_id, deploy_unit_id)

    is_ready = responses[0]
    return is_ready == "true"


def get_stage_status(stage_id, yp_token):
    import yt.yson
    from yt.orm.library.common import NoSuchObjectError
    from yp.client import YpClient

    yp_client = YpClient("xdc", config={"token": yp_token})
    try:
        yson_objects = yp_client.get_object("stage", stage_id, selectors=["/status"])
    except NoSuchObjectError:
        raise DeployServiceNotFound(stage_id)
    return yt.yson.convert.yson_to_json(yson_objects)


def get_pod_statuses(stage_id, deploy_unit_id, cluster, yp_token):
    import yt.yson
    from yt.orm.library.common import NoSuchObjectError
    from yp.client import YpClient

    yp_client = YpClient(cluster, config={"token": yp_token})
    try:
        yson_objects = yp_client.select_objects(
            "pod",
            filter="[/meta/pod_set_id]=\"{}.{}\"".format(stage_id, deploy_unit_id),
            selectors=["/status"],
        )
    except NoSuchObjectError:
        raise DeployServiceNotFound(stage_id)
    return yt.yson.convert.yson_to_json(yson_objects)


def remove_stage(stage_id, yp_token, force=False):
    from yt.orm.library.common import NoSuchObjectError
    from yp.client import YpClient

    yp_client = YpClient("xdc", config={"token": yp_token})
    try:
        yp_client.remove_object("stage", stage_id)
    except NoSuchObjectError as e:
        if force:
            logger.debug("Stage %s does not exist", stage_id)
            return
        raise e


def get_stage_url(stage_id):
    """Get URL for Deploy stage

    :param stage_id: Deploy stage id
    :type stage_id: str
    :return: Deploy stage URL
    :rtype: str
    """
    return "https://deploy.yandex-team.ru/stages/{}".format(stage_id)


def get_stage_errors(stage_status):
    errors = []
    try:
        for deploy_unit_id, deploy_unit_data in stage_status[0]["deploy_units"].items():
            for cluster_id, cluster_data in deploy_unit_data["replica_set"]["cluster_statuses"].items():
                succeeded = (
                    cluster_data["status"]["deploy_status"]["details"]["controller_status"]
                    ["last_attempt"]["succeeded"])
                if succeeded["status"] == "false":
                    errors.append(
                        "deploy_unit: {}\ncluster: {}\nerror:\n{}".format(
                            deploy_unit_id, cluster_id, succeeded["reason"]))
    except (KeyError, IndexError) as e:
        logger.error("Failed to parse stage status: %s", e)

    return errors


def get_pod_errors(pod_statuses):
    errors = []
    for pod_status in pod_statuses:
        try:
            for workload in pod_status[0]["agent"]["pod_agent_payload"]["status"]["workloads"]:
                if workload["state"] == "active":
                    continue
                errors.append(
                    "pod: {pod_id}\nworkload: {workload_id}\nfailure reason: {failure_reason}"
                    .format(
                        pod_id=pod_status[0]["agent"]["pod_agent_payload"]["status"]["id"],
                        workload_id=workload["id"],
                        failure_reason=workload["start"]["last"]["fail_reason"],
                    )
                )
        except (KeyError, IndexError) as e:
            logger.error("Failed to parse pod statuses: %s", e)

    return errors
