import six


def update_url_query(path, query, append=True):
    """Update URL query

    :param query: list of name, value pairs
    :type query: list[tuple]
    :param append: If `append` is True, append items to url's query. Otherwise, overwrite existing items with the same name.
    :type append: bool
    :return: URL with updated query part
    :rtype: str
    """
    if not query:
        return path

    url = six.moves.urllib.parse.urlsplit(path)

    updated_query = six.moves.urllib.parse.parse_qsl(url.query, keep_blank_values=True)
    if not append:
        update_keys = set(key for key, _ in query)
        updated_query = list(filter(lambda pair: pair[0] not in update_keys, updated_query))
    updated_query += query

    updated_query_str = six.moves.urllib.parse.urlencode(updated_query, doseq=True)

    return six.moves.urllib.parse.urlunsplit((url.scheme, url.netloc, url.path, updated_query_str, url.fragment))


class HTTPRequest(six.moves.BaseHTTPServer.BaseHTTPRequestHandler):
    """Class to parse, store and serialize raw http request"""
    def __init__(self, request_string):
        self.rfile = six.BytesIO(request_string)
        self.raw_requestline = self.rfile.readline()
        self.parse_request()
        self.data = self.rfile.read()

    def update_query(self, query):
        """Update self.path query

        :param query: list of name, value pairs
        :type query: list[tuple]
        """
        self.path = update_url_query(self.path, query)

    def serialize(self):
        """Serialize to raw http request

        :return: raw http request
        :rtype: (bytes|str)
        """
        result = "{command} {path} {version}\r\n".format(command=self.command, path=self.path, version=self.request_version)
        for header_name, header_value in self.headers.items():
            result += "{key}: {value}\r\n".format(key=header_name, value=header_value)
        result += "\r\n"
        result = result.encode() + self.data

        return result
