import datetime
import json
import logging
from collections import namedtuple

import requests

from sandbox.projects.common.nanny.client import NannyClient


logger = logging.getLogger(__name__)


NANNY_REQUEST_TIMEOUT = 60
ISS_HOOK_INSTALL_MAX_EXECUTION_TIME = int(datetime.timedelta(hours=1).total_seconds())


def generate_info_attrs(category, description, abc_service_id, labels):
    return {
        "category": category,
        "desc": description,
        "labels": labels,
        "abc_group": abc_service_id,
        "recipes": {
            "content": [{
                "desc": "Activate",
                "labels": [],
                "id": "default",
                "context": [
                    {
                        "key": "prepare_operating_degrade_level",
                        "value": "1"
                    },
                    {
                        "key": "operating_degrade_level",
                        "value": "1"
                    },
                    {
                        "key": "stop_degrade_level",
                        "value": "0.25"
                    },
                    {
                        "key": "max_degrade_speed",
                        "value": "1"
                    },
                ],
                "name": "_activate_service_configuration.yaml"
            }]
        }
    }


def generate_runtime_attrs_for_nanny_service(sandbox_files=(), static_files=()):
    return {
        "instance_spec": {
            "layersConfig": {
                "layer": [
                    {
                        "url": [
                            "rbtorrent:b583dac42e6a45f123e08dcc37ad34f1db99f165"
                        ],
                        "fetchableMeta": {
                            "type": "SANDBOX_RESOURCE",
                            "sandboxResource": {
                                "taskId": "408053428",
                                "resourceId": "904144736"
                            }
                        }
                    }
                ],
            },
            # Setup unistat monitoring
            # https://wiki.yandex-team.ru/runtime-cloud/nanny/howto-manage-service/#yasm-monitoring-rtcsmall
            "containers": [
                {
                    "name": "daemon",
                    "unistatEndpoints": [
                        {
                        "path": "admin?action=stat",
                        "port": "{BSCONFIG_IPORT}"
                        }
                    ],
                    "env": [{
                        "name": "MKL_CBWR",
                        "valueFrom": {
                            "literalEnv": {
                                "value": "COMPATIBLE"
                            },
                            "type": "LITERAL_ENV"
                        }
                    }]
                }
            ],
            # Setup logrotate
            # https://wiki.yandex-team.ru/runtime-cloud/nanny/howtos/logrotate/
            "auxDaemons": [{
                "type": "LOGROTATE"
            }],
            # Enable host skynet
            # https://wiki.yandex-team.ru/skynet/drafts/edinyjj-xostovyjj-skynet/
            "hostProvidedDaemons": [{
                "type": "HOST_SKYNET"
            }],
        },
        "instances": {
            "chosen_type": "YP_POD_IDS",
            "iss_settings": {
                "hooks_time_limits": {
                    "iss_hook_install": {
                        "max_execution_time": ISS_HOOK_INSTALL_MAX_EXECUTION_TIME,
                    },
                },
            },
        },
        "engines": {
            "engine_type": "YP_LITE"
        },
        "resources": {
            "sandbox_files": sandbox_files,
            "static_files": static_files,
        }
    }


def update_cleanup_policy(nanny_token, service_id, snapshots_count=2, disposable_count=0, stalled_ttl="PT24H"):
    import nanny_rpc_client
    from nanny_repo import repo_api_stub, repo_api_pb2, repo_pb2

    client = nanny_rpc_client.RetryingRpcClient(
        "https://nanny.yandex-team.ru/api/repo",
        oauth_token=nanny_token,
        request_timeout=NANNY_REQUEST_TIMEOUT)
    repo_stub = repo_api_stub.RepoServiceStub(client)

    res = repo_stub.get_cleanup_policy(repo_api_pb2.GetCleanupPolicyRequest(policy_id=service_id))

    # https://bb.yandex-team.ru/projects/NANNY/repos/nanny/browse/nanny/src/protobuf/nanny_repo/repo_api.proto#111,439
    req = repo_api_pb2.UpdateCleanupPolicyRequest(
        meta=repo_pb2.CleanupPolicyMeta(
            id=service_id,
            version=res.policy.meta.version,
        ),
        spec=repo_pb2.CleanupPolicySpec(
            type=0,  # SIMPLE_COUNT_LIMIT
            simple_count_limit=repo_pb2.CleanupPolicySimpleCountLimit(
                snapshots_count=snapshots_count,
                disposable_count=disposable_count,
                stalled_ttl=stalled_ttl,
            )
        )
    )

    return repo_stub.update_cleanup_policy(req)


PersistentVolume = namedtuple("PersistentVolume", [
    "disk_quota_gigabytes",
    "mount_point",
    "storage_class",
    "bandwidth_guarantee_megabytes_per_sec",
    "bandwidth_limit_megabytes_per_sec",
])


AllocationRequest = namedtuple("AllocationRequest", [
    "replicas",
    "snapshots_count",
    "root_volume_storage_class",
    "root_fs_quota_gigabytes",
    "work_dir_quota_gigabytes",
    "root_bandwidth_guarantee_megabytes_per_sec",
    "root_bandwidth_limit_megabytes_per_sec",
    "vcpu_guarantee",
    "memory_guarantee_megabytes",
    "network_macro",
    "network_bandwidth_guarantee_megabytes_per_sec",
    "network_bandwidth_limit_megabytes_per_sec",
    "persistent_volumes",
])


def create_pod_set(
        nanny_token,
        service_id,
        cluster,
        allocation_request,
        abc_service_id=None,
):
    from library.python import retry

    import nanny_rpc_client
    from infra.nanny.yp_lite_api.proto import pod_sets_api_pb2
    from infra.nanny.yp_lite_api.py_stubs import pod_sets_api_stub

    client = nanny_rpc_client.RetryingRpcClient(
        "https://yp-lite-ui.nanny.yandex-team.ru/api/yplite/pod-sets/",
        oauth_token=nanny_token,
        request_timeout=NANNY_REQUEST_TIMEOUT,
        # For idempotent processing of operations https://wiki.yandex-team.ru/runtime-cloud/nanny/yp-lite-api/#caveats
        req_id_header="X-Req-Id")
    stub = pod_sets_api_stub.YpLiteUIPodSetsServiceStub(client)

    allocation_request_dict = allocation_request._asdict()
    del allocation_request_dict['persistent_volumes']
    allocation_request_obj = pod_sets_api_pb2.AllocationRequest(**allocation_request_dict)

    for volume in allocation_request.persistent_volumes:
        allocation_request_obj.persistent_volumes.add(
            disk_quota_gigabytes=volume.disk_quota_gigabytes,
            mount_point=volume.mount_point,
            storage_class=volume.storage_class,
            bandwidth_guarantee_megabytes_per_sec=volume.bandwidth_guarantee_megabytes_per_sec,
            bandwidth_limit_megabytes_per_sec=volume.bandwidth_limit_megabytes_per_sec,
        )

    req = pod_sets_api_pb2.CreatePodSetRequest(
        service_id=service_id,
        cluster=cluster.upper(),
        quota_settings=pod_sets_api_pb2.ResourceQuotaSettings(
            # ABC_SERVICE = 0;
            # TMP_ACCOUNT = 1;
            # https://a.yandex-team.ru/arc/trunk/arcadia/infra/nanny/yp_lite_api/proto/pod_sets_api.proto?rev=r8528453#L260
            mode=abc_service_id is None,
            abc_service_id=abc_service_id
        ),
        antiaffinity_constraints=pod_sets_api_pb2.AntiaffinityConstraints(
            node_max_pods=1,
        ),
        allocation_request=allocation_request_obj,
        node_affinity=pod_sets_api_pb2.NodeAffinity(
            require_avx=1,
            require_intel=1,
        )
    )
    logger.debug("CreatePodSetRequest: %s", req)

    # Pod set creation can fail with timeout
    # Example: InternalError: HTTPSConnectionPool(host="...", port=8443): Read timed out. (read timeout=0.2)
    retry_conf = retry.DEFAULT_CONF.on(nanny_rpc_client.exceptions.InternalError)
    return retry.retry_call(stub.create_pod_set, [req], conf=retry_conf)


def create_endpoint_set(nanny_token, service_id, cluster, endpoint_set_id, protocol="tcp", port=80):
    import nanny_rpc_client
    from infra.nanny.yp_lite_api.proto import endpoint_sets_pb2, endpoint_sets_api_pb2
    from infra.nanny.yp_lite_api.py_stubs import endpoint_sets_api_stub

    client = nanny_rpc_client.RetryingRpcClient(
        "https://yp-lite-ui.nanny.yandex-team.ru/api/yplite/endpoint-sets/",
        oauth_token=nanny_token,
        request_timeout=NANNY_REQUEST_TIMEOUT)
    stub = endpoint_sets_api_stub.YpLiteUIEndpointSetsServiceStub(client)

    req = endpoint_sets_api_pb2.CreateEndpointSetRequest(
        cluster=cluster.upper(),
        meta=endpoint_sets_pb2.EndpointSetMeta(
            id=endpoint_set_id,
            service_id=service_id,
        ),
        spec=endpoint_sets_pb2.EndpointSetSpec(
            protocol=protocol,
            port=port,
        )
    )

    return stub.create_endpoint_set(req)


def add_pods_to_nanny_service(nanny_client, service_id, pod_ids, cluster, instance_tags):
    runtime_attrs = nanny_client.get_service_runtime_attrs(service_id)["content"]
    instances_section = runtime_attrs["instances"]
    instances_section["yp_pod_ids"] = {
        "pods": [
            {
                "cluster": cluster.upper(),
                "pod_id": pod_id
            }
            for pod_id in pod_ids
        ],
        "orthogonal_tags": instance_tags
    }
    return nanny_client.update_service_runtime_attrs_section(service_id, "instances", instances_section)


def get_staff_group_id(abc_service_id, auth_token):
    from library.python.retry import retry

    @retry()
    def _get_staff_group_id(abc_service_id, auth_token):
        r = requests.get(
            "https://staff-api.yandex-team.ru/v3/groups",
            headers={"Authorization": "OAuth {}".format(auth_token)},
            params={"_query": "service.id=={}".format(abc_service_id)})
        if not r.ok:
            raise Exception(r.text)
        return r.json()["result"][0]["id"]

    return _get_staff_group_id(abc_service_id, auth_token)


def create_hamster_service(
        nanny_token,
        staff_token,
        service_id,
        service_labels,
        cluster,
        category,
        abc_service_id,
        allocation_request,
        instance_tags,
        endpoint_set_id,
        admin_logins=(),
        admin_abc_services=(),
        sandbox_files=(),
        static_files=(),
        description="-",
        activate=True,
        use_tmp_quota=False,
):
    nanny_client = NannyClient(
        api_url="http://nanny.yandex-team.ru/",
        oauth_token=nanny_token,
    )

    info_attrs = generate_info_attrs(category, description, abc_service_id, service_labels)

    admin_groups = [
        str(get_staff_group_id(admin_abc_service_id, staff_token))
        for admin_abc_service_id in admin_abc_services
    ]

    auth_attrs = {
        "conf_managers": {
            "groups": admin_groups,
            "logins": admin_logins,
        },
        "ops_managers": {
            "groups": admin_groups,
            "logins": admin_logins
        },
        "owners": {
            "groups": admin_groups,
            "logins": admin_logins
        }
    }

    runtime_attrs = generate_runtime_attrs_for_nanny_service(sandbox_files, static_files)

    logger.debug(
        "Create nanny service: %s", json.dumps({
            "service_id": service_id,
            "info_attrs": info_attrs,
            "auth_attrs": auth_attrs,
            "runtime_attrs": runtime_attrs,
        }, indent=2),
    )
    nanny_client.create_service(service_id, info_attrs, auth_attrs=auth_attrs, runtime_attrs=runtime_attrs)

    # Set number of cleanup policy PREPARED/CREATED snapshots to 1
    # to set total snapshots count as small as posible to save the YP quote.
    # The smallest number of snapshots is 3:
    # cleanup policy PREPARED snapshots limit + ACTIVATING snapshot + DEACTIVATE_PENDING snapshot.
    # For details see https://st.yandex-team.ru/SWAT-6345
    logger.debug("Update cleanup policies")
    update_cleanup_policy(nanny_token, service_id, snapshots_count=1)

    logger.debug("Create pod set")
    response = create_pod_set(
        nanny_token, service_id, cluster, allocation_request,
        abc_service_id=None if use_tmp_quota else abc_service_id)
    pod_ids = response.pod_ids

    logger.debug("Add pods %s to service", pod_ids)
    response = add_pods_to_nanny_service(nanny_client, service_id, pod_ids, cluster, instance_tags)
    snapshot_id = response["runtime_attrs"]["_id"]
    logger.debug("Created snapshot %s", snapshot_id)

    logger.debug("Create endpoint set with id=%s", endpoint_set_id)
    create_endpoint_set(nanny_token, service_id, cluster, endpoint_set_id)

    if activate:
        logger.debug("Activate service")
        nanny_client.set_snapshot_state(service_id, snapshot_id, "ACTIVE", comment="-", recipe="default")

    return snapshot_id
