import logging

import requests

from sandbox.projects.common.nanny.client import NannyClient, NannyApiException


logger = logging.getLogger(__name__)


def remove_pod_set(nanny_token, service_id, cluster):
    import nanny_rpc_client
    from infra.nanny.yp_lite_api.proto import pod_sets_api_pb2
    from infra.nanny.yp_lite_api.py_stubs import pod_sets_api_stub
    from library.python import retry

    client = nanny_rpc_client.RetryingRpcClient(
        "https://yp-lite-ui.nanny.yandex-team.ru/api/yplite/pod-sets/",
        request_timeout=60,
        oauth_token=nanny_token)
    stub = pod_sets_api_stub.YpLiteUIPodSetsServiceStub(client)

    req = pod_sets_api_pb2.RemovePodSetRequest(
        service_id=service_id,
        cluster=cluster.upper(),
    )

    retry_conf = retry.DEFAULT_CONF\
        .waiting(100, backoff=1.1)\
        .upto(minutes=10)\
        .on(nanny_rpc_client.exceptions.BadRequestError, nanny_rpc_client.exceptions.InternalError)
    try:
        retry.retry_call(stub.remove_pod_set, [req], conf=retry_conf)
    except nanny_rpc_client.exceptions.NotFoundError as e:
        logger.error(e)


def remove_endpoint_sets(nanny_token, service_id, cluster):
    import nanny_rpc_client
    from infra.nanny.yp_lite_api.proto import endpoint_sets_api_pb2
    from infra.nanny.yp_lite_api.py_stubs import endpoint_sets_api_stub
    from library.python import retry

    client = nanny_rpc_client.RetryingRpcClient(
        "https://yp-lite-ui.nanny.yandex-team.ru/api/yplite/endpoint-sets/",
        oauth_token=nanny_token)
    stub = endpoint_sets_api_stub.YpLiteUIEndpointSetsServiceStub(client)

    req = endpoint_sets_api_pb2.ListEndpointSetsRequest(
        service_id=service_id,
        cluster=cluster.upper(),
    )
    res = stub.list_endpoint_sets(req)
    logger.debug("Endpoint sets: %s", res.endpoint_sets)

    retry_conf = retry.DEFAULT_CONF\
        .waiting(10, backoff=1.5)\
        .upto(minutes=10)\
        .on(requests.exceptions.ReadTimeout)
    for endpoint_set in res.endpoint_sets:
        req = endpoint_sets_api_pb2.RemoveEndpointSetRequest(
            id=endpoint_set.meta.id,
            version=endpoint_set.meta.version,
            cluster=cluster.upper(),
        )
        logger.debug("Remove endpoint set: %s", req)
        retry.retry_call(stub.remove_endpoint_set, [req], conf=retry_conf)


def remove_nanny_service(nanny_token, service_id, cluster, force=False):
    nanny_client = NannyClient(
        api_url="http://nanny.yandex-team.ru/",
        oauth_token=nanny_token,
    )

    logger.debug("Shutdown service")
    try:
        nanny_client.shutdown(service_id)
    except NannyApiException as e:
        if force and e.http_error is not None and e.http_error.response.status_code == requests.codes.not_found:
            logger.warn("Service %s not found", service_id)
            return

    logger.debug("Delete all snapshots")
    nanny_client.delete_all_snapshots(service_id)

    logger.debug("Remove podset")
    remove_pod_set(nanny_token, service_id, cluster)

    logger.debug("Remove endpoint sets")
    remove_endpoint_sets(nanny_token, service_id, cluster)

    logger.debug("Remove service")
    nanny_client.delete_service(service_id)

    logger.debug("Service removed: %s", service_id)
