import logging

import requests

from sandbox.projects.common.nanny.client import NannyClient, NannyApiException


logger = logging.getLogger(__name__)


class NannyServiceNotFound(Exception):
    """Raised when Nanny service does not exist
    """


def get_active_snapshot_id(nanny_token, service_id):
    """Get active snapshot for Nanny service.

    :param nanny_token: Nanny API token
    :type nanny_token: str
    :param service_id: Nanny service id
    :type service_id: str
    :raises NannyServiceNotFound: If service does not exist
    :return: Active snapshot id or None if unknown error occurred
    :rtype: str|None
    """    """"""
    nanny_client = NannyClient(api_url="http://nanny.yandex-team.ru/", oauth_token=nanny_token)

    try:
        return nanny_client.get_service_active_runtime_attrs(service_id)["_id"]
    except NannyApiException as nanny_exception:
        if nanny_exception.http_error.response.status_code == requests.codes.not_found:
            raise NannyServiceNotFound(service_id)

        logger.error(nanny_exception)
        return None


def is_service_active(nanny_token, service_id):
    """Check if Nanny service is active.

    :param nanny_token: Nanny API token
    :type nanny_token: str
    :param service_id: Nanny service id
    :type service_id: str
    :raises NannyServiceNotFound: If service does not exist
    :return: Is Nanny service active
    :rtype: bool
    """
    return get_active_snapshot_id(nanny_token, service_id) is not None


def get_service_url(service_id):
    """Get Nanny service URL.

    :param service_id: Nanny service id
    :type service_id: str
    :return: Nanny service URL
    :rtype: str
    """
    return "https://nanny.yandex-team.ru/ui/#/services/catalog/{}/".format(service_id)
