import enum
import json

from sandbox import sdk2
from sandbox.projects.yabs.qa.resource_types import YabsServerExtServiceEndpoint


class ExternalServiceType(enum.Enum):
    NANNY = 'nanny'
    DEPLOY = 'deploy'


class UnknownExternalServiceType(Exception):
    pass


def create_hamster_spec_resource(cluster, endpoint_set, service_id, service_type, service_tag, meta_env=None, **attrs):
    """Create resource with spec for external service's hamster

    :param cluster: Cluster name
    :type cluster: str
    :param endpoint_set: Endpointset id
    :type endpoint_set: str
    :param service_id: Service id for Nanny, stage id for Deploy
    :type service_id: str
    :param service_type: Service type, one of "nanny" or "deploy"
    :type service_type: ExternalServiceType
    :param service_tag: Service tag
    :type service_tag: str
    :param meta_env: Custom environment for meta
    :type meta_env: dict
    :return: Spec resource
    :rtype: sdk2.Resource
    """
    spec = {
        "cluster": cluster.lower(),
        "endpoint_set": endpoint_set,
        "service_id": service_id,
        "service_type": service_type.value,
    }
    if meta_env:
        spec["meta_env"] = meta_env

    spec_filename = "{}_spec.json".format(service_tag)
    with open(spec_filename, "w") as f:
        json.dump(spec, f)

    resource = YabsServerExtServiceEndpoint(
        task=sdk2.Task.current,
        description="External service spec for {}".format(service_tag),
        path=spec_filename,
        service_tag=service_tag,
        **attrs
    )

    sdk2.ResourceData(resource).ready()
    return resource
