import difflib
import logging
import textwrap
import socket
from collections import namedtuple

import requests as rq
import six

from sandbox import sdk2
from sandbox.projects.common.testenv_client import TEClient
from sandbox.projects.common.yabs.server.util import truncate_output_parameters
from sandbox.projects.common.yql import run_query
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootStability import YabsServerB2BFuncShootStability
from sandbox.projects.yabs.qa.utils.general import get_resource_html_hyperlink
from sandbox.projects.yabs.qa.hamster.utils import calculate_enabled_hamsters


logging.basicConfig(level=logging.DEBUG)
logger = logging.getLogger(__name__)
logging.getLogger("urllib3").setLevel(logging.WARN)
logging.getLogger("yql").setLevel(logging.WARN)


TE_DATABASE = 'yabs-2.0'


def get_ext_requests_from_yt(yql_token, service_tag, limit=1, proxy="hahn"):
    import yt

    query = textwrap.dedent("""\
        -- https://yql.yandex-team.ru/docs/yt/faq/temp
        PRAGMA yt.TmpFolder = "//home/yabs-cs-sandbox/tmp/yql_tmp_folder";
        PRAGMA yt.QueryCacheTtl = "1d";

        $last_hourly_log = (
            SELECT
                Max(Path)
            FROM
                FOLDER("//logs/bs-proto-extstat-log/1h")
            WHERE
                Type = "table"
        );

        SELECT
            Request
        FROM
            CONCAT($last_hourly_log)
        WHERE
            Tag == "{tag}"
            AND Request is not null
        LIMIT {limit};
    """.format(limit=limit, tag=service_tag))
    request = run_query(
        query_id='get requests to {}'.format(service_tag),
        query=query,
        yql_token=yql_token,
        db=proxy,
        wait=False,
        syntax_version=1)
    logger.debug('Run YQL query %s', request.share_url)

    if request.errors:
        for error in request.errors:
            logging.error(error)
        raise Exception("YQL request failed")

    result = []
    for table in request.get_results():
        table.fetch_full_data()
        for row in table.rows:
            result.append(yt.yson.get_bytes(row[0]))
    return result


def resolve_endpoints(endpoint_set_id, cluster):
    from infra.yp_service_discovery.python.resolver.resolver import Resolver
    from infra.yp_service_discovery.api import api_pb2

    resolver = Resolver(client_name='test:{}'.format(socket.gethostname()), timeout=5)

    request = api_pb2.TReqResolveEndpoints()
    request.cluster_name = cluster.lower()
    request.endpoint_set_id = endpoint_set_id

    result = resolver.resolve_endpoints(request)
    return [endpoint.fqdn for endpoint in result.endpoint_set.endpoints]


def get_response(endpoint, http_request):
    url = six.moves.urllib.parse.urlunsplit(("http", endpoint, http_request.path, "", ""))
    logger.debug("Request url: %s", url)

    try:
        r = rq.request(
            method=http_request.command,
            url=url,
            headers=http_request.headers,
            data=http_request.data,
        )
    except rq.exceptions.RequestException as e:
        logger.warning("%s not responding: %s", endpoint, e)
        return None

    if not r.ok:
        logger.warning("%s responded with %s, %s", endpoint, r.status_code, r.text)
        return None

    return r.text


ResponseDiff = namedtuple("ResponseDiff", ("request_id", "host_a", "host_b", "diff"))


def check_service_stability(endpoint_set_id, cluster, http_requests):
    endpoints = resolve_endpoints(endpoint_set_id, cluster)
    logger.debug("Endpoints: %s", endpoints)

    if len(endpoints) <= 1:
        return []

    response_diffs = []
    for request_id, request in enumerate(http_requests):
        responses = []
        for endpoint in endpoints:
            response = get_response(endpoint, request)
            if response is None:
                continue
            responses.append((endpoint, response))

        canonical_endpoint, canonical_response = responses[0]

        for endpoint, response in responses[1:]:
            if response is None:
                continue

            if canonical_response != response:
                diff_lines = difflib.unified_diff(
                    canonical_response.splitlines(True),
                    response.splitlines(True),
                    fromfile=canonical_endpoint,
                    tofile=endpoint,
                )
                response_diffs.append(ResponseDiff(
                    request_id=request_id,
                    host_a=canonical_endpoint,
                    host_b=endpoint,
                    diff="".join(diff_lines)
                ))

    return response_diffs


def run_stability_shoot(task, testenv_job_name, ext_service_tag, ext_service_endpoint_resource_id):
    last_succeeded_tasks = TEClient.get_last_sandbox_task_ids(TE_DATABASE, [testenv_job_name], success=True)
    if not last_succeeded_tasks:
        raise Exception("No succeeded tasks found for job {}".format(testenv_job_name))
    baseline_task_data = last_succeeded_tasks[0]
    logger.debug("Baseline task: %s", baseline_task_data)

    baseline_task = sdk2.Task[baseline_task_data["task_id"]]
    baseline_task_parameters = dict(baseline_task.Parameters)

    enabled_hamsters = calculate_enabled_hamsters(
        baseline_task_parameters.get("hamster_ext_service_tags", []),
        [int(resource.id) for resource in baseline_task_parameters.get("ext_service_endpoint_resources", [])]
    )
    enabled_hamsters[ext_service_tag] = ext_service_endpoint_resource_id

    shoot_stability_task_parameters = truncate_output_parameters(
        baseline_task_parameters,
        baseline_task.type.Parameters,
    )
    shoot_stability_task_parameters.update(
        ext_service_endpoint_resources=list(enabled_hamsters.values()),
        hamster_ext_service_tags=list(enabled_hamsters.keys()),
        stability_runs=2,
    )

    stability_shoot_task = YabsServerB2BFuncShootStability(
        task,
        tags=task.Parameters.tags,
        hints=list(task.hints),
        description=(
            "Check yabs-server stability with \"{tag}\" hamster {resource}"
            .format(
                tag=ext_service_tag,
                resource=get_resource_html_hyperlink(ext_service_endpoint_resource_id),
            )
        ),
        **shoot_stability_task_parameters
    ).enqueue()
    return stability_shoot_task.id
