import logging

from sandbox.common.rest import Client as SandboxRestClient
from sandbox.projects.yabs.qa.hamster.check_readiness import (
    check_hamster_readiness,
    HamsterNotFound,
)
from sandbox.projects.yabs.qa.hamster.deploy import get_stage_url as get_deploy_stage_url
from sandbox.projects.yabs.qa.hamster.nanny.utils import get_service_url as get_nanny_service_url
from sandbox.projects.yabs.qa.hamster.spec import ExternalServiceType, UnknownExternalServiceType
from sandbox.projects.yabs.qa.utils.resource import (
    get_resource_attributes,
    json_from_resource,
)


logger = logging.getLogger(__name__)


class AmbiguousEndpointResources(Exception):
    """Used to indicate ambiguous hamster endpoint resources."""


def calculate_enabled_hamsters(hamster_tags, resource_ids):
    """Calculate enabled hamsters.
    Hamster is considered enabled if its tag is presented in `hamster_tags`
    and there is exactly one resource with id from `resource_ids`
    and with `service_tag` attribute equal to hamster service tag.

    :param hamster_tags: External services tags
    :type hamster_tags: list[str]
    :param resource_ids: Resource ids
    :type resource_ids: list[int]
    :return: Dict from enabled hamster to resource id
    :rtype: dict[str, int]
    :raises TypeError: if items in `resource_ids` are not integers.
    :raises AmbiguousEndpointResources: if there are more than one resource with the same tag.
    """
    tag_to_resource = {}
    for resource_id in resource_ids:
        if not isinstance(resource_id, int):
            raise TypeError("Resource id must be int but {} is given: {}".format(type(resource_id), resource_id))

        resource_attrs = get_resource_attributes(resource_id)
        try:
            resource_service_tag = resource_attrs["service_tag"]
        except KeyError:
            logger.warning("Resource %s has no \"service_tag\" attribute", resource_id)
            continue
        if resource_service_tag not in hamster_tags:
            continue

        if resource_service_tag in tag_to_resource:
            raise AmbiguousEndpointResources(
                "Multiple resources with attribute \"service_tag\"={service_tag}: {resource_ids}"
                .format(
                    service_tag=resource_service_tag,
                    resource_ids=[tag_to_resource[resource_service_tag], resource_id]
                )
            )
        tag_to_resource[resource_service_tag] = resource_id

    return tag_to_resource


def shoot_task_enabled_hamsters(task):
    """Calculate enabled hamsters for shoot task.
    Basically it's just syntactic sugar for calling calculate_enabled_hamsters() with sandbox task object

    :param task: Sandbox task
    :type task: sdk2.Task
    :return: Dict from enabled hamster to resource id
    :rtype: dict[str, int]
    """
    try:
        parameters = task.MutableParameters
    except AttributeError:
        parameters = task.Parameters
    return calculate_enabled_hamsters(
        parameters.hamster_ext_service_tags,
        [int(resource.id) for resource in parameters.ext_service_endpoint_resources or []],
    )


def get_hamster_url(resource_id):
    """Get URL for hamster service by endpoint resource.

    :param resource_id: Endpoint resource id
    :type resource_id: int
    :raises UnknownExternalServiceType: If service type is unknown
    :return: Hamster URL
    :rtype: str
    """
    endpoint_data = json_from_resource(resource_id)
    service_id = endpoint_data["service_id"]
    service_type = endpoint_data["service_type"]

    if service_type == ExternalServiceType.NANNY.value:
        return get_nanny_service_url(service_id)

    if service_type == ExternalServiceType.DEPLOY.value:
        return get_deploy_stage_url(service_id)

    raise UnknownExternalServiceType(service_type)


def get_bad_hamsters(task_id, nanny_token, yp_token, task_parameters=None):
    sandbox_client = SandboxRestClient()

    task_parameters = task_parameters or sandbox_client.task[task_id].read()["input_parameters"]

    enabled_hamsters = calculate_enabled_hamsters(
        task_parameters.get("hamster_ext_service_tags") or [],
        map(int, task_parameters.get("ext_service_endpoint_resources") or []),
    )
    logger.debug("Enabled hamsters in sandbox task %s: %s", task_id, enabled_hamsters)

    bad_hamsters = {}
    for service_tag, resource_id in enabled_hamsters.items():
        try:
            hamster_is_ready = check_hamster_readiness(resource_id, nanny_token, yp_token)
        except HamsterNotFound:
            bad_hamsters[service_tag] = resource_id
        except ImportError:
            logger.exception("ImportError caught, probably running in non-binary mode")
        else:
            if not hamster_is_ready:
                bad_hamsters[service_tag] = resource_id

    return bad_hamsters
