import subprocess
import uuid
import os
import signal

from sandbox.projects.yabs.qa.module_base import ModuleBase
from sandbox.projects.yabs.qa.sut.components.base import demangle_perf_file
from sandbox.projects.common.yabs.server.util.general import CustomAssert

from sandbox.sdk2.helpers import ProcessLog
from sandbox.common.errors import TaskFailure


class PerfModule(ModuleBase):
    def __init__(self, adapter):
        ModuleBase.__init__(self, adapter)
        self._flamegraph_bundle_path = self.adapter.get_flamegraph_bundle_path()
        self._perf_process = None

    def start_perf_record(self, process_pid):
        CustomAssert(self._perf_process is None, 'Concurrent perf recording for several processes with a single module instance is not supported', TaskFailure)
        data_path = str(uuid.uuid4()) + '.perf_record'
        cmdline = [
            'perf',
            'record',
            '-p', str(process_pid),
            '-g',
            '-F', str(self.adapter.get_perf_sample_frequency()),
            '-o', data_path
        ]
        self._process_log_context = ProcessLog(self.adapter.task_instance, 'collect_perf_data')
        self._process_log_context.__enter__()
        self._perf_process = subprocess.Popen(cmdline, stdout=self._process_log_context.stdout, stderr=subprocess.STDOUT)
        return data_path

    def stop_perf_record(self):
        self._perf_process.send_signal(signal.SIGINT)
        self._perf_process.communicate()
        self._perf_process = None
        self._process_log_context.__exit__(None, None, None)

    class PerfRecordContext(object):
        def __init__(self, module, process_pid):
            self._module = module
            self._process_pid = process_pid

        def __enter__(self):
            self._data_path = self._module.start_perf_record(self._process_pid)
            return self._data_path

        def __exit__(self, *args):
            self._module.stop_perf_record()
            demangle_perf_file(self._data_path)

    def collect_perf_data(self, process_pid):
        return self.PerfRecordContext(self, process_pid)

    def run_perf_pipeline_tool(self, cmdline, out_path):
        with open(out_path, 'w') as outfile, ProcessLog(self.adapter.task_instance, 'perf_pipeline') as process_log:
            subprocess.Popen(
                cmdline,
                stdout=outfile,
                stderr=process_log.stderr,
            ).wait()
        return out_path

    def generate_perf_results(self, data_path, processed_perf_results_path=None):
        if processed_perf_results_path is None:
            processed_perf_results_path = str(uuid.uuid4()) + '.perf_results'
            os.makedirs(processed_perf_results_path)
        perf_script_out_path = self.run_perf_pipeline_tool([
            'perf',
            'script',
            '-i', data_path
        ], out_path=str(uuid.uuid4()) + '.script')
        stack_collapsed_path = self.run_perf_pipeline_tool([
            'perl',
            os.path.join(self._flamegraph_bundle_path, 'stackcollapse-perf.pl'),
            perf_script_out_path
        ], out_path=os.path.join(processed_perf_results_path, 'perf_stack_collapsed'))
        stack_collapsed_path = self.run_perf_pipeline_tool([
            'perl',
            os.path.join(self._flamegraph_bundle_path, 'flamegraph.pl'),
            '--hash',
            stack_collapsed_path
        ], out_path=os.path.join(processed_perf_results_path, 'perf_flamegraph.svg'))
        self.run_perf_pipeline_tool([
            'perf',
            'report',
            '-i', data_path
        ], out_path=os.path.join(processed_perf_results_path, 'perf_top'))
        return processed_perf_results_path

    class PerfRecordAndProcessContext(object):
        def __init__(self, module, process_pid):
            self._module = module
            self._process_pid = process_pid

        def __enter__(self):
            self._data_path = self._module.start_perf_record(self._process_pid)
            self._processed_perf_results_path = str(uuid.uuid4()) + '.perf_results'
            os.makedirs(self._processed_perf_results_path)
            return self._processed_perf_results_path

        def __exit__(self, *args):
            self._module.stop_perf_record()
            demangle_perf_file(self._data_path)
            self._module.generate_perf_results(self._data_path, self._processed_perf_results_path)

    def collect_and_process_perf_data(self, process_pid):
        return self.PerfRecordAndProcessContext(self, process_pid)
