import itertools

from sandbox.projects.yabs.qa.utils import Contextable
from sandbox.projects.common.yabs.performance_best_cmp.stats import hodges_lehmann_median

from collections import namedtuple
from sandbox.projects.common import utils2


class RpsResults(namedtuple('RpsResults', [
    'rps_percentile_80',
    'rps_hl_median',
    'rps_list',
]), Contextable):
    pass


RpsResults.__new__.__defaults__ = (0,) * len(RpsResults._fields)


class RpsResultsCorrected(namedtuple('RpsResultsCorrected', [
    'rps_percentile_80_corrected',
    'rps_hl_median_corrected',
    'rps_list_corrected',
]), Contextable):
    pass


RpsResultsCorrected.__new__.__defaults__ = (0,) * len(RpsResultsCorrected._fields)


class PerfResults(namedtuple('PerfResults', [
    'perf_resource_id',
    'perf_link',
]), Contextable):
    pass


PerfResults.__new__.__defaults__ = (0, '')


class EngineMode(object):
    meta = 'meta'
    stat = 'stat'


class ResultType(object):
    baseline = 'baseline'
    increase = 'increase'


def get_rss_result_key(result_type, engine_mode, hl_median=False):
    return 'rss_{result_type}{hl_median}_{engine_mode}'.format(
        result_type=result_type,
        engine_mode=engine_mode,
        hl_median='_hl_median' if hl_median else ''
    )


class RssResults(namedtuple('RssResults', [
    get_rss_result_key(result_type, engine_mode, hl_median=hl_median)
    for result_type in (ResultType.baseline, ResultType.increase)
    for engine_mode in (EngineMode.meta, EngineMode.stat)
    for hl_median in (True, False)
]), Contextable):
    pass


RssResults.__new__.__defaults__ = (0,) * len(RssResults._fields)


RssResultsSingle = namedtuple('RssResultsSingle', [
    'rss_baseline',
    'rss_baseline_hl_median',
    'rss_increase',
    'rss_increase_hl_median'
])
RssResultsSingle.__new__.__defaults__ = (0,) * len(RssResultsSingle._fields)


def get_rss_results_meta(results):
    return RssResultsSingle(**{key: getattr(results, key + '_meta') for key in RssResultsSingle._fields})


def get_rss_results_stat(results):
    return RssResultsSingle(**{key: getattr(results, key + '_stat') for key in RssResultsSingle._fields})


def collect_rps(subtasks, corrected=False, second_batch=False):
    def get_rps(task):
        if corrected:
            if second_batch:
                return task.Context.rps_corrected_2
            else:
                return task.Context.rps_corrected
        else:
            if second_batch:
                return task.Context.rps_2
            else:
                return task.Context.rps

    rps_list_sorted = sorted((get_rps(subtask) for subtask in subtasks), reverse=True)
    rps_percentile_80 = rps_list_sorted[(len(rps_list_sorted) / 5)]
    rps_hl_median = hodges_lehmann_median(rps_list_sorted)

    return (RpsResultsCorrected if corrected else RpsResults)(
        rps_percentile_80,
        rps_hl_median,
        rps_list_sorted
    )


def collect_avg_rps(subtasks, corrected=False, second_batch=False):
    def get_rps(task):
        if corrected:
            if second_batch:
                return task.Context.rps_corrected_list_2
            else:
                return task.Context.rps_corrected_list
        else:
            if second_batch:
                return task.Context.rps_list_2
            else:
                return task.Context.rps_list

    rps_list_sorted = sorted(itertools.chain(*(get_rps(subtask) for subtask in subtasks)), reverse=True)
    if not rps_list_sorted:
        return (RpsResultsCorrected if corrected else RpsResults)(
            0,
            0,
            rps_list_sorted
        )

    rps_percentile_80 = rps_list_sorted[(len(rps_list_sorted) / 5)]
    rps_hl_median = hodges_lehmann_median(rps_list_sorted)

    return (RpsResultsCorrected if corrected else RpsResults)(
        rps_percentile_80,
        rps_hl_median,
        rps_list_sorted
    )


def collect_rss(subtasks, second_batch=False):
    baselines = {}
    increases = {}
    for task in subtasks:
        for mode in ('stat', 'meta'):
            context_key = 'usage_data_{}{}'.format(mode, ('_2' if second_batch else ''))
            for impact in getattr(task.Context, context_key):
                start = impact['start']["non_HP_maps_anon_total"]
                stop = impact['stop']["non_HP_maps_anon_total"]
                (baselines.setdefault(mode, [])).append(start)
                (increases.setdefault(mode, [])).append(stop - start)

    result_dict = {}
    for data, kind in (baselines, ResultType.baseline), (increases, ResultType.increase):
        for mode, values in data.iteritems():
            result_dict[get_rss_result_key(kind, mode, hl_median=False)] = values
            result_dict[get_rss_result_key(kind, mode, hl_median=True)] = hodges_lehmann_median(values)

    return RssResults(**result_dict)


def collect_perf(subtasks, second_batch=False):
    if not second_batch:
        max_perf_subtask = max(subtasks, key=lambda subtask: (subtask.Context.rps if subtask.Context.perf_resource_id else -1))
        perf_resource_id = max_perf_subtask.Context.perf_resource_id
    else:
        max_perf_subtask = max(subtasks, key=lambda subtask: (subtask.Context.rps_2 if subtask.Context.perf_resource_id_2 else -1))
        perf_resource_id = max_perf_subtask.Context.perf_resource_id_2
    if not perf_resource_id:
        return PerfResults()
    perf_link = utils2.resource_redirect_url(perf_resource_id)
    return PerfResults(
        perf_resource_id=perf_resource_id,
        perf_link=perf_link
    )
