from __future__ import division

import os
import uuid
import logging
import subprocess
from collections import namedtuple

from sandbox import sdk2
from sandbox import common

from sandbox.common.types.task import ReleaseStatus
from sandbox.projects.common import utils2
from sandbox.projects.common.yabs.performance_best_cmp.stats import hodges_lehmann_diff
from sandbox.projects.resource_types import FLAME_GRAPH_BUNDLE
from sandbox.projects.yabs.qa.performance import collect
from sandbox.sandboxsdk.paths import get_logs_folder


CompareRssResults = namedtuple('CompareRssResults', ['pre_hl_median', 'test_hl_median', 'diff_percent', 'engine_mode'])
CompareRpsResults = namedtuple('CompareRpsResults', ['pre_hl_median', 'test_hl_median', 'diff_percent'])
ComparePerfResults = namedtuple('ComparePerfResults', ['diff_flamegraph_url'])


def compare_rps(pre_results, test_results):
    rps_hl_diff = hodges_lehmann_diff(pre_results.rps_list, test_results.rps_list)
    return CompareRpsResults(
        pre_hl_median=pre_results.rps_hl_median,
        test_hl_median=test_results.rps_hl_median,
        diff_percent=100 * rps_hl_diff / pre_results.rps_hl_median,
    )


def compare_rss(pre_results, test_results, meta_mode=True):
    pre_results = collect.get_rss_results_meta(pre_results) if meta_mode else collect.get_rss_results_stat(pre_results)
    test_results = collect.get_rss_results_meta(test_results) if meta_mode else collect.get_rss_results_stat(test_results)
    rss_hl_diff = hodges_lehmann_diff(pre_results.rss_increase, test_results.rss_increase)

    return CompareRssResults(
        pre_hl_median=pre_results.rss_increase_hl_median,
        test_hl_median=test_results.rss_increase_hl_median,
        diff_percent=100 * rss_hl_diff / pre_results.rss_increase_hl_median,
        engine_mode=('Meta' if meta_mode else 'Stat'),
    )


def compare_rss_stat(*args, **kwargs):
    return compare_rss(meta_mode=False, *args, **kwargs)


def compare_perf(task, pre_results, test_results):
    if not pre_results.perf_resource_id or not test_results.perf_resource_id:
        return
    flame_bundle_path = str(sdk2.ResourceData(sdk2.Resource.find(FLAME_GRAPH_BUNDLE, attrs={'released': ReleaseStatus.STABLE}).order(-sdk2.Resource.id).first()).path)
    pre_perf_stack_path = os.path.join(str(sdk2.ResourceData(sdk2.Resource[pre_results.perf_resource_id]).path), 'perf_stack_collapsed')
    test_perf_stack_path = os.path.join(str(sdk2.ResourceData(sdk2.Resource[test_results.perf_resource_id]).path), 'perf_stack_collapsed')
    stack_diff_path = str(uuid.uuid4()) + '.diff_stack'
    with open(stack_diff_path, 'w') as out_file, sdk2.helpers.ProcessLog(task, logger=logging.getLogger('perf_stack_diff')) as process_log_context:
        subprocess.Popen(
            [
                'perl',
                os.path.join(flame_bundle_path, 'difffolded.pl'),
                pre_perf_stack_path,
                test_perf_stack_path
            ],
            stdout=out_file,
            stderr=process_log_context.stderr
        ).wait()
    diff_flamegraph_path = os.path.join(get_logs_folder(), str(uuid.uuid4()) + '.diff.svg')
    with open(diff_flamegraph_path, 'w') as out_file, sdk2.helpers.ProcessLog(task, logger=logging.getLogger('diff_flamegraph_path')) as process_log_context:
        subprocess.Popen(
            [
                'perl',
                os.path.join(flame_bundle_path, 'flamegraph.pl'),
                stack_diff_path
            ],
            stdout=out_file,
            stderr=process_log_context.stderr
        ).wait()
    log_resource_id = common.rest.Client().resource.read(task_id=task.id,
                                                         type='TASK_LOGS',
                                                         limit=10,
                                                         order='-id')['items'][0]['id']
    diff_flamegraph_url = utils2.resource_redirect_url(log_resource_id, os.path.basename(diff_flamegraph_path))
    return ComparePerfResults(diff_flamegraph_url=diff_flamegraph_url)


def dict_metrics_diff(pre_metrics, test_metrics):
    diff = {}
    for key in set().union(pre_metrics.keys(), test_metrics.keys()):
        diff_value = test_metrics.get(key, 0) - pre_metrics.get(key, 0)
        metric_diff = {
            'pre': pre_metrics.get(key, 0),
            'test': test_metrics.get(key, 0),
            'diff': diff_value,
            'change': (100 * diff_value / pre_metrics.get(key, 0)) if pre_metrics.get(key, 0) else float('nan'),
        }
        template = '{change:+.1f}% ({diff:+.3f}, {pre:.3f} -> {test:.3f})'
        if isinstance(pre_metrics.get(key, 0), int) and isinstance(test_metrics.get(key, 0), int):
            template = '{change:+.1f}% ({diff:+}, {pre} -> {test})'
        diff[key] = template.format(**metric_diff)
    return diff
