#!/usr/bin/env python2

"""yabs-server-phantom access log parser command line"""

import itertools as itt
import logging
import logging.handlers
import socket
import multiprocessing

# pylint: disable=R0903


class MetricsWriter(object):
    """Abstarct class taht writes metrics somewhere"""

    def write(self, role, timestamp, metrics):
        """Write or send metrics somewhere"""
        raise NotImplementedError("Class is abstract")


class GraphiteSenderClient(MetricsWriter):
    """
    Sends metrics via graphite-sender for this host and specified role
    """

    def __init__(self, prefix='one_sec', dry_graphite=False, debug_graphite=False, addrs=('localhost:42000',)):
        """
        :param str prefix: added before fqdn into each metric
        :param bool debug_graphite: write metrics to debug log
        """
        self.dry_graphite = dry_graphite
        self.debug_graphite = debug_graphite
        self.prefix = prefix + '.' + socket.getfqdn().replace('.', '_')
        self.addrs = []
        for addr in addrs:
            host, _, port = addr.partition(':')
            self.addrs.append((host, int(port)))

    def write(self, role, timestamp, metrics):
        """Aggregate metrics and return string to be sent"""
        logging.debug(
            "Dumping %s metrics for %s with timestamp %s",
            len(metrics), role, timestamp
        )

        metrics_str = '\n'.join('%s.%s.%s %s %s' % (self.prefix, role, key, metrics[key], timestamp) for key in sorted(metrics))

        if self.debug_graphite:
            logging.debug("Sending metrics:\n%s", metrics_str)
        if self.dry_graphite:
            return
        for addr in self.addrs:
            try:
                sck = socket.create_connection(addr)
                sck.sendall(metrics_str)
                sck.close()
            except Exception as exc:  # pylint: disable=W0703
                msg = str(exc)
                if msg:
                    logging.error("Failed to send data to %s:%s (%s)", addr[0], addr[1], msg)
                else:
                    logging.exception("Failed to send data to %s:%s", addr[0], addr[1])


class MetricFile(MetricsWriter):
    """
    Used by LogHandlers to write metrics into file
    """

    def __init__(self, filename, rotate=True):
        self.metrics_log = logging.getLogger(filename)
        self.metrics_log.setLevel(logging.INFO)
        if rotate:
            metric_handler = logging.handlers.RotatingFileHandler(
                filename, maxBytes=16 * 1024 * 1024, backupCount=4)
        else:
            metric_handler = logging.FileHandler(filename)
        metric_handler.setFormatter(logging.Formatter("%(message)s"))
        self.metrics_log.addHandler(metric_handler)
        self.metrics_log.propagate = False

        self.lock = multiprocessing.Lock()

    def write(self, role, timestamp, metrics):
        """
        Writes metric line in tskv format:
        Example: role=metapartner\tts=12345678\tmetric1=value1\tmetric2=value2..
        :param dict metrics:
        """
        metric_iter = itt.chain([('role', role), ('ts', timestamp)], metrics.iteritems())
        metric_line = '\t'.join('='.join((key, str(val))) for key, val in metric_iter)
        with self.lock:
            self.metrics_log.info(metric_line)
