#!/usr/bin/env python2
# pylint:disable=C0103

"""yabs-server-phantom access log parser command line"""

import logging
import logging.handlers
import argparse
import multiprocessing
import signal
import time

from sandbox.projects.yabs.qa.performance.stats import access, monitor, common


def parse_options():
    '''parse command line options'''
    parser = argparse.ArgumentParser()
    parser.add_argument("--debug",
                        action='store_true',
                        help='Debug logging',
                        default=False,
                        dest='debug'
                        )
    parser.add_argument("--no-monitor",
                        action='store_true')
    parser.add_argument("--no-access-log",
                        action='store_true')

    parser.add_argument("--dry-graphite",
                        action='store_true',
                        help='Do not send metrics to graphite',
                        default=False,
                        dest='dry_graphite'
                        )
    parser.add_argument("--debug-graphite",
                        action='store_true',
                        help='Debug logging with graphite metrics in log',
                        default=False,
                        dest='debug_graphite'
                        )
    parser.add_argument("--metrics-log", "--mlog",
                        help='Output file for metrics',
                        default=None,
                        dest='metrics_log_path'
                        )
    parser.add_argument("--log-pattern",
                        help='Pattern for matching log files',
                        default="/var/log/yabs/access-*.log",
                        dest='log_glob'
                        )
    parser.add_argument("--quantiles",
                        help='Quantiles',
                        default="90, 95, 98, 99, 99.5, 99.8",
                        dest='quantile_str'
                        )
    parser.add_argument("--disable-sharded-ext",
                        help=("Do not calculate per-shard (i.e. per-io_client) "
                              "statistics for specified ext requests "
                              "(comma-separated list)"),
                        default='bsbts_all, market, yacofast, count, yt',
                        dest='disable_ext_sharded_str'
                        )
    parser.add_argument("--max-lag",
                        help="Drop lines with timestamp lagging by more than "
                             "specified number of seconds (default: 60)",
                        type=int,
                        default=60,
                        dest='max_lag'
                        )

    return parser.parse_args()


def _wake_on_sigchld(sig, frame):  # pylint: disable=W0613
    """One-time SIGCHLD handler"""
    pass


def main():
    """
    Set up logging
    and run handlers in child processes
    """
    opts = parse_options()

    logger = logging.getLogger()
    logger.setLevel(logging.DEBUG if opts.debug or opts.debug_graphite else logging.INFO)

    handler = logging.StreamHandler()
    handler.setFormatter(
        logging.Formatter("%(asctime)s [%(levelname)s] (%(name)s) %(message)s"))
    logger.addHandler(handler)

    metrics_writers = []
    if not opts.dry_graphite or opts.debug_graphite:
        metrics_writers.append(common.GraphiteSenderClient(
            dry_graphite=opts.dry_graphite,
            debug_graphite=opts.debug_graphite
            ))
    if opts.metrics_log_path is not None:
        metrics_writers.append(common.MetricFile(opts.metrics_log_path))

    handlers = {}
    if not opts.no_access_log:
        handlers['access'] = access.MultiLogHandler(
            log_glob=opts.log_glob,
            max_lag=opts.max_lag,
            quantiles=set(float(part.strip())
                          for part in opts.quantile_str.split(',')),
            disable_ext_sharded=set(
                part.strip()
                for part in opts.disable_ext_sharded_str.split(',')),
            metrics_writers=metrics_writers
        )
    if not opts.no_monitor:
        handlers['monitor'] = monitor.MultiMonitorHandler(metrics_writers)

    signal.signal(signal.SIGCHLD, _wake_on_sigchld)

    processes = []
    for name, handler in handlers.iteritems():
        prc = multiprocessing.Process(
            target=handler.run,
            name=name
        )
        prc.daemon = True,
        processes.append(prc)
        prc.start()

    while all(prc.is_alive() for prc in processes):
        time.sleep(10)
    for prc in processes:
        prc.terminate()
    exitcode = 0
    for prc in processes:
        prc.join()
        exitcode = prc.exitcode or exitcode
    return exitcode


if __name__ == '__main__':
    exit(main())
