#!/usr/bin/env python2
# pylint: disable=W0703
# pylint: disable=R0903
"""
Poll monitoring approximately once per second.
Calculate phantom schedulers utilization during the period
passed since the previous poll.
NOTE:
We do not rely on arrival times of monitoring responses
when calculating utilization.
"""
import urllib
import json
import time
import logging
import logging.handlers
import subprocess
import errno


class ThreadPoolStats(object):

    def __init__(self, pool_data):
        self.busy = 0.0
        self.total = 0.0
        for entry in pool_data.itervalues():
            tstate = entry['tstate']
            total = sum(tstate.itervalues())
            self.total += total
            self.busy += (total - tstate['idle'])
        self.busy /= len(pool_data)
        self.total /= len(pool_data)

    def get_metric_value(self, prev):
        return (self.busy - prev.busy) / (self.total - prev.total)


def _get_scheduler_stats(name, data):
    result = dict()
    if 'pool' in data:
        result['utilization.%s_pool' % name] = ThreadPoolStats(data['pool'])
    if 'bqs' in data:
        result['utilization.%s' % name] = ThreadPoolStats(data['bqs'])
    return result


class ModeHandler(object):
    """
    Handles monitoring of specified yabs-server-phantom mode
    """

    def __init__(self, mode, port, metrics_writers):
        self.metrics_writers = metrics_writers
        self.prev_stats = {}
        self.mode = mode
        self.port = port
        self.url = "http://127.0.0.1:%s" % port

    def next_metrics(self):
        """
        Get next metrics from monitoring data
        """
        timestamp = int(time.time())
        try:
            mon_str = urllib.request.urlopen(self.url, timeout=10).read()
        except Exception:
            logging.error("Failed to obtain monitoring data from monitor2 port")
            raise
        mon_data = json.loads(mon_str)
        stats = dict()
        for obj, obj_data in mon_data.iteritems():
            if obj.split('_')[-1] == 'scheduler':
                stats.update(_get_scheduler_stats(obj, obj_data))

        metrics = dict()
        for metric in stats.viewkeys() & self.prev_stats.viewkeys():
            metrics['phantom_stat.' + metric] = stats[metric].get_metric_value(self.prev_stats[metric])

        self.prev_stats = stats
        for wrt in self.metrics_writers:
            wrt.write(self.mode, timestamp, metrics)


class MultiMonitorHandler(object):
    """
    Handles all yabs-server-phantom modes at this host
    """

    def __init__(self, metrics_writers, ports_file, poll_interval=1, refresh_interval=10):
        self.mode_handlers = {}
        self.metrics_writers = metrics_writers
        self.ports_file = ports_file
        self.poll_interval = poll_interval
        self.refresh_interval = refresh_interval

    def run(self):
        """Run loop"""
        next_refresh_time = 0
        while True:
            now = time.time()
            next_time = now + self.poll_interval

            if now > next_refresh_time:
                self._refresh_handlers()
                next_refresh_time = now + self.refresh_interval

            for mode, handler in self.mode_handlers.iteritems():
                try:
                    handler.next_metrics()
                except Exception:
                    logging.exception("Failed to handle next metrics from %s:", mode)
                    next_refresh_time = 0

            time.sleep(max(next_time - time.time(), 0))

    def _refresh_handlers(self):
        """Create handlers for new modes, delete handlers taht disappeared"""
        try:
            mode_ports = self._get_mode_ports()
        except Exception:
            logging.exception("Failed to obtain modes and ports:")
            return

        for mode, handler in self.mode_handlers.iteritems():
            if mode not in mode_ports or mode_ports[mode] != handler.port:
                logging.info("Stopping to handle %s", mode)
                del self.mode_handlers[mode]

        for mode in mode_ports.viewkeys() - self.mode_handlers.viewkeys():
            try:
                logging.info("Adding handler for %s", mode)
                self.mode_handlers[mode] = ModeHandler(mode, mode_ports[mode], self.metrics_writers)
            except Exception:
                logging.exception("Failed to create handler for %s:", mode)

    def _get_mode_ports(self):
        try:
            with open(self.ports_file) as pf:
                ports = json.load(pf)
        except IOError as err:
            if err.errno == errno.ENOENT:
                pass
            logging.warning("Failed to read %s:", self.ports_file, exc_info=True)
        except Exception:
            logging.warning("Failed to read %s:", self.ports_file, exc_info=True)
        else:
            return {mode: mode_ports['monitor2'] for mode, mode_ports in ports.iteritems()}

        modes = set(open('/var/log/yabs/modes_list').read().split())
        ports_by_mode = dict()
        for mode in modes:
            port = subprocess.check_output(['/usr/bin/yabs-server-port', mode, 'monitor2'])
            port = int(port)
            logging.info("monitor2 port for %s is %s", mode, port)
            ports_by_mode[mode] = port

        return ports_by_mode
