from sandbox import sdk2

from sandbox.common.types import resource as ctr

# for backward compatibility, do not use it, use sdk2.Resource instead
# FIXME switch them all to sdk2.Resource

DEFAULT_TTL = 42
DEFAULT_AMMO_TTL = 7
AbstractResource = sdk2.resource.AbstractResource


class BaseBackupSdk2Resource(sdk2.Resource):
    auto_backup = True


class BaseBackupAbstractResource(sdk2.resource.AbstractResource):
    auto_backup = True


class UCTool(BaseBackupSdk2Resource):
    """ Resource with uc tool """
    name = "UC_TOOL"
    releasable = True
    ttl = 'inf'


class MYSQL_TABLE_ARCHIVE(BaseBackupAbstractResource):
    """
        Archive with MySql table
        Attributes: instance=(yabs|yabsst17|...), db=(yabsdb|yabsinfo), table=(Resource|Banner113|...), date=%Y%m%d
    """
    auto_backup = True
    sync_upload_to_mds = True
    ttl = 28


class YABS_SERVER_B2B_SPEC(BaseBackupAbstractResource):
    """
        Json spec for local debug of yabs-server
    """


class YABS_SERVER_BASE_PACK_EXECUTABLE(BaseBackupAbstractResource):
    # FIXME BSSERVER-6343
    """
    Executable to compress/decompress yabs-server binary bases
    """
    releasable = True


class LINEAR_MODEL_DUMPS(BaseBackupAbstractResource):
    # FIXME ditch smokes!
    """
        Archive with linear model dumps
        Attributes: date=%Y%m%d, shard={01..12} or 0 for meta
    """
    auto_backup = True
    sync_upload_to_mds = True
    ttl = 7


class YABS_MYSQL_RESTORE_DESCRIPTION(BaseBackupAbstractResource):
    """
        Json that describes what mysql tables to restore (and how).
        Example:
        {
            "yabs.yabsdb.Constant": "bsdb",
            "yabsst01.yabsdb.Constant": "=yabs.yabsdb.Constant", # create symlink
            "yabsst01.yabsdb.PhrasePrice": "bstat01",
            "yabs.yabsuinfo.QQQ": "bsdb"
        }
    """
    auto_backup = True
    sync_upload_to_mds = True
    ttl = 28
    # YabsServerGetSQLArchive is virtually the only task that creates this.
    # restart_policy=IGNORE is needed for this task to work correctly.
    restart_policy = ctr.RestartPolicy.IGNORE


class YABS_MYSQL_ARCHIVE_CONTENTS(BaseBackupAbstractResource):
    """
        Json that describes how to find prepared MYSQL tables in Sandbox.
        Example:
        {
            "yabs.yabsdb.Constant": 12345678, # resource id
            "yabsst01.yabsdb.Constant": "=yabs.yabsdb.Constant", # create symlink
            "yabsst01.yabsdb.PhrasePrice": 13456789,
            "yabs.yabsuinfo.QQQ": 14567890 # resource id
        }
    """
    auto_backup = True
    sync_upload_to_mds = True
    ttl = 28


class YABS_CS_SAMPLING_TABLES(BaseBackupAbstractResource):
    """
        This resource is created manually and contains dict of input archive tables to be sampled with parameters used to form sampling YQL query here:
        https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/yabs/qa/bases/sample_tables/run.py

        Json structure of this resource should be the same as TABLES_TO_SAMPLE here:
        https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/yabs/qa/bases/sample_tables/tables.py
    """
    auto_backup = True
    sync_upload_to_mds = True
    ttl = 'inf'


class YABS_CS_SAMPLING_QUERY_TEMPLATE(BaseBackupAbstractResource):
    """
        This resource is created manually and is used as YQL template for sampling of input archive in:
        https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/yabs/qa/bases/sample_tables/run.py

        Example of YQL template: https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/yabs/qa/bases/sample_tables/templates/sample_tables.sql
    """
    auto_backup = True
    sync_upload_to_mds = True
    ttl = 'inf'


class YABS_CS_INPUT_SPEC(BaseBackupAbstractResource):
    """
        Input spec for cs_import (usually table paths only)
    """
    auto_backup = True
    sync_upload_to_mds = True
    ttl = 28


class YABS_REPORT_RESOURCE(BaseBackupAbstractResource):
    """
       resources for b2b report
    """
    auto_backup = True
    sync_upload_to_mds = True
    releasable = True
    ttl = 28


class YABS_REPORT_RESOURCE_DEPENDENCIES(BaseBackupAbstractResource):
    """pre-installed npm dependencies for yabs/qa/compare_report_viewer
    """
    auto_backup = True
    sync_upload_to_mds = True
    releasable = True
    ttl = 28


class YABS_SERVER_REQUEST_LOG_GZ(BaseBackupAbstractResource):
    """
        Request log from yabs-server-phantom.
        Attributes: type=(yabs|bs|rank|...), date=%Y%m%d, hour=14
    """
    auto_backup = True
    sync_upload_to_mds = True
    ttl = DEFAULT_AMMO_TTL


class YABS_SERVER_DOLBILKA_PLAN(BaseBackupAbstractResource):
    """
        Dolbilka plan for shooting yabs-server meta*
        Attributes: type=(yabs|bs|rank|...), date=%Y%m%d, hour=14
    """
    auto_backup = True
    sync_upload_to_mds = True
    cachedaemon_dump_res_id = sdk2.Attributes.Integer("cachedaemon dump resource id")
    subtype = sdk2.Attributes.String()
    testenv_switch_trigger = sdk2.Attributes.String()
    handlers = sdk2.Attributes.String()
    ttl = DEFAULT_AMMO_TTL


class YABS_SERVER_FT_DOLBILKA_PLAN(BaseBackupAbstractResource):
    """
        Dolbilka plan for shooting yabs-server meta*
        Attributes: type=(yabs|bs|rank|...), date=%Y%m%d, hour=14
    """
    auto_backup = True
    restart_policy = ctr.RestartPolicy.DELETE
    sync_upload_to_mds = True
    ttl = DEFAULT_AMMO_TTL


class YABS_SERVER_REQUEST_LOAD_AMMO(BaseBackupAbstractResource):
    # FIXME bad name for current use case
    """
        Ammofile for shooting yabs-server-phantom via yandextank
        Attributes:
            all attributes of YABS_SERVER_REQUEST_LOG_GZ
            plus subtype=('http'|'https'|'dsp'|...)
    """
    auto_backup = True
    sync_upload_to_mds = True
    ttl = 7
    shoot_index = sdk2.Attributes.Integer('Content type')


class YABS_SERVER_TESTENV_DB_FLAGS(BaseBackupAbstractResource):
    """
        Resource with flags for testenv databases of yabs-server.
        Do not autogenerate this resource!
    """
    auto_backup = True
    sync_upload_to_mds = True
    ttl = 999


class YABS_SERVER_TESTENV_SHARD_MAP(BaseBackupAbstractResource):
    # FIXME Description
    """
        Resource with mapping test->shard no.
        Mapping is stored as resource attributes.
        Do not store data inside!
    """
    auto_backup = True
    sync_upload_to_mds = True
    ttl = 365  # Yes, 1 year


class YABS_SERVER_PERFORMANCE_BANLIST(BaseBackupAbstractResource):
    # FIXME Remove altogether
    """
        Client banlist for YABS_SERVER_*_PERFORMANCE tasks
    """
    auto_backup = True
    sync_upload_to_mds = True
    ttl = 720


class YABS_SERVER_CACHE_DAEMON_STUB_DATA(BaseBackupAbstractResource):
    """
    Dump of cachedaemon filesystem cache + cachedaemon configs used in yabs-server tests.
    Primary use is a stub in yabs-server tests.
    """
    auto_backup = True
    sync_upload_to_mds = True
    cache_daemon_res_id = sdk2.Attributes.Integer("CACHE_DAEMON resource id", required=True)
    provided_tags = sdk2.Attributes.String('External tags provided by stub', required=False)
    key_header = sdk2.Attributes.String('Key header', required=False)
    testenv_switch_trigger = sdk2.Attributes.String('Testenv switch trigger for new resource switch sequence')
    ttl = DEFAULT_AMMO_TTL


class YABS_SERVER_B2B_BINARY_BASE(BaseBackupAbstractResource):
    """
        Compressed binary base for yabs-server-phantom b2b tests
        Attributes:
            type = st001, dbe, dblm1_0, etc...
            ver = DB_VER in yabs-server: 1234567, etc...
            date (in YYYYMMDD format)
            origin = prod|sandbox (sandbox if not specified)
            pack = ypk|gz (gz if not specified)
            source = production|trunk|oneshots|<unknown>
    """
    auto_backup = True
    sync_upload_to_mds = True
    any_arch = False
    ttl = 7
    tag = sdk2.Attributes.String('Base tag')
    db_ver = sdk2.Attributes.Integer('Basever')
    pack = sdk2.Attributes.String('Pack type')
    source = sdk2.Attributes.String('Base source')
    unpacked_size = sdk2.Attributes.Integer('Unpacked size in bytes')


class YABS_RELEASE_BASE_RESOURCE(BaseBackupAbstractResource):
    releasable = True
    any_arch = False
    auto_backup = True
    sync_upload_to_mds = True
    releasers = ['YABS_SERVER_SANDBOX_TESTS']
    restart_policy = ctr.RestartPolicy.DELETE
    revision = sdk2.Attributes.Integer('Base revision')
    ttl = 14
    major_version = sdk2.Attributes.Integer('Major version', default=0)
    minor_version = sdk2.Attributes.Integer('Minor version', default=0)


class BS_RELEASE_YT(YABS_RELEASE_BASE_RESOURCE):
    pass


class BS_RELEASE_TAR(YABS_RELEASE_BASE_RESOURCE):
    pass


class BS_RELEASE_DEBIAN(YABS_RELEASE_BASE_RESOURCE):
    pass


class YABS_UDF(YABS_RELEASE_BASE_RESOURCE):
    pass


class YABS_ZELIBOBA_MODELS_EXECUTABLE(YABS_RELEASE_BASE_RESOURCE):
    releasers = ['YABS_SERVER_SANDBOX_TESTS', 'ulyanin', 'grievous']


class YABS_HIT_MODELS_EXECUTABLE(YABS_RELEASE_BASE_RESOURCE):
    releasers = ['YABS_SERVER_SANDBOX_TESTS', 'SEARCH-RELEASERS']


class YabsServerGenStatStub(BaseBackupSdk2Resource):
    """yabs/server/tools/gen_stat_stub binary

    Deprecated. See BSSERVER-23013 for more info
    """
    auto_backup = True
    sync_upload_to_mds = True
    releasable = True
    releasers = ['YABS_SERVER_SANDBOX_TESTS']


class YabsServerStatStub(BaseBackupSdk2Resource):
    """Stat stubs resource (cachedaemon with real stats responses)"""
    auto_backup = True
    sync_upload_to_mds = True
    releasers = ["YABS_SERVER_SANDBOX_TESTS"]


class YabsServerProductionAnalyzerTool(BaseBackupSdk2Resource):
    """Resource with production analyzer tool binary"""
    releasable = True
    auto_backup = True
    sync_upload_to_mds = True


class YabsServerProductionAnalyzerFlameGraphReport(BaseBackupSdk2Resource):
    """Resource for production analyzer tool flame graph report"""
    releasable = True
    auto_backup = True
    sync_upload_to_mds = True


class YabsServerProductionAnalyzerReport(BaseBackupSdk2Resource):
    """Resource for production analyzer tool report"""
    releasable = True
    auto_backup = True
    sync_upload_to_mds = True


class YabsServerProductionAnalyzerHtmlReport(BaseBackupSdk2Resource):
    """Resource for production analyzer tool HTML report"""
    releasable = True
    auto_backup = True
    sync_upload_to_mds = True


class MkdbInfoResource(BaseBackupSdk2Resource):
    """Stores mkdb_info result"""
    releasable = True
    auto_backup = True
    sync_upload_to_mds = True
    releasers = ["YABS_SERVER_SANDBOX_TESTS"]
    ttl = DEFAULT_TTL


class YabsResponseDumpUnpacker(BaseBackupAbstractResource):
    """ Resource with new dolbilo2json utility """
    releasable = True
    auto_backup = True
    sync_upload_to_mds = True
    releasers = ["YABS_SERVER_SANDBOX_TESTS"]
    ttl = DEFAULT_TTL


class YabsRuntimeAuditDaemon(BaseBackupSdk2Resource):
    """Serves md5 hash of any accessible file"""
    releasable = True
    auto_backup = True
    sync_upload_to_mds = True
    releasers = ["YABS_SERVER_SANDBOX_TESTS"]


class YABS_SERVER_CPUS_PER_ROLE(BaseBackupSdk2Resource):
    """
    Numbers of CPUs used for yabs, bs, bsrank in production
    """
    pass


class YABS_SERVER_BASE_CHKDB(BaseBackupSdk2Resource):
    """
    Yabs_chkdb of server bases
    """
    auto_backup = True
    sync_upload_to_mds = True
    ttl = 30


class YABS_SERVER_BASE_CHKDB_WITHOUT_HASH(BaseBackupSdk2Resource):
    """
    Yabs_chkdb of server bases without hash
    """
    auto_backup = True
    sync_upload_to_mds = True
    ttl = 30


class YT_ONESHOT_BINARY(BaseBackupSdk2Resource):
    """
    YT oneshot binary
    """
    auto_backup = True
    sync_upload_to_mds = True
    oneshot_path = sdk2.Attributes.String('Oneshot path relative to arcadia root')
    ttl = DEFAULT_TTL


class YT_ONESHOT_OUTPUT(BaseBackupSdk2Resource):
    """
    YT oneshot output
    """
    auto_backup = True
    sync_upload_to_mds = True
    ttl = 30


class YT_ONESHOTS_PACKAGE(YABS_RELEASE_BASE_RESOURCE):
    """
    YT common oneshots package
    """
    pass


class YabsDevelopmentYtOneshotBinary(YT_ONESHOT_BINARY):
    releasers = [
        'igorock',
        'demurin'
    ]
    releasable = True


class YABS_SERVER_YT_DIGEST(BaseBackupAbstractResource):
    """
        yabs/server/tools/yt_digest
    """
    auto_backup = True
    sync_upload_to_mds = True
    releasable = True
    releasers = ['YABS_SERVER_SANDBOX_TESTS']


class YABS_CS_SETTINGS_ARCHIVE(BaseBackupSdk2Resource):
    """
    Archive with production content system settings
    """
    auto_backup = True
    sync_upload_to_mds = True
    ttl = 30
    yt_proxy = sdk2.Attributes.String('yt_proxy')
    testenv_switch_trigger = sdk2.Attributes.String('testenv_switch_trigger')


class YABS_CS_SETTINGS_PATCH(BaseBackupSdk2Resource):
    """
    Jsondiff (syntax=compact) patch to update archive with prod cs settings
    """
    auto_backup = True
    sync_upload_to_mds = True
    ttl = 30


class YabsSamplingQueryTemplate(BaseBackupSdk2Resource):
    """ Jinja2 template with YQL query for sampling tables
    """
    auto_backup = True
    sync_upload_to_mds = True
    ttl = 30


class YabsSamplingTables(BaseBackupSdk2Resource):
    """ JSON dict with sampling tables config.
    key: relative path to table
    value: dict with config
    """
    auto_backup = True
    sync_upload_to_mds = True
    ttl = 30


class YabsSamplingTablesBlackList(BaseBackupSdk2Resource):
    """ JSON list with black list of tables for sampling.
    """
    auto_backup = True
    sync_upload_to_mds = True
    ttl = 30


class YabsDiffReport(BaseBackupSdk2Resource):
    auto_backup = True
    sync_upload_to_mds = True
    test_name = sdk2.Attributes.String("Test name")
    testenv_db = sdk2.Attributes.String("TestEnv database", default="yabs-2.0")
    arcanum_review_id = sdk2.Attributes.Integer('Review', default=0)
    restart_policy = ctr.RestartPolicy.DELETE
    has_diff = sdk2.Attributes.Bool("Resource contains actual diff", default=False)


class YabsFtSmartMetaReport(YabsDiffReport):
    """ Resource with smart report metadata """
    meta_mode = sdk2.Attributes.String("BS meta mode", required=True)


class YabsDbSmartMetaReport(YabsDiffReport):
    """ Resource with diff report """
    shard_key = sdk2.Attributes.String("Db shard key")


class MR_UTILS(BaseBackupSdk2Resource):
    """ads/quality/adv_machine/tools/mr_utils"""
    auto_backup = True
    sync_upload_to_mds = True
    ttl = 30
    releasable = True
    releasers = ['YABS_SERVER_SANDBOX_TESTS']


class YabsBsteViewer(BaseBackupSdk2Resource):
    auto_backup = True
    sync_upload_to_mds = True
    releasable = True
    releasers = ['YABS_SERVER_SANDBOX_TESTS']


class YabsBsteUpdater(BaseBackupSdk2Resource):
    auto_backup = True
    sync_upload_to_mds = True
    releasable = True
    releasers = ['YABS_SERVER_SANDBOX_TESTS']


class YABS_CS_IMPORT_OUT_LOCATION(BaseBackupAbstractResource):
    """ Input spec for cs_import (usually table paths only) """
    auto_backup = True
    sync_upload_to_mds = True
    ttl = 28


class YabsMysqlLxcImage(sdk2.service_resources.LxcContainer):
    """ LXC sandbox image with yabs mysql, dbrestore and experiment-switcher installed """
    auto_backup = True
    sync_upload_to_mds = True
    ttl = 28
    releasable = True
    releasers = ['YABS_SERVER_SANDBOX_TESTS']

    __default_attribute__ = sdk2.parameters.String


class YABS_SERVER_META_NETWORK_STATISTICS(BaseBackupSdk2Resource):
    """
        Resource with network statistics for meta by handlers
        {
            Ext: {
                Handler: {
                    "request_size": int,
                    "response_size": int,
                    "count": int,
                }
            }
        }
    """
    auto_backup = True
    sync_upload_to_mds = True
    ttl = 28


class YABS_SERVER_SAAS_FREEZE_DATA(BaseBackupSdk2Resource):
    """SaaS active frozen snapshot data

    Example:
    {
        "id": "20210318T133536Z",
        "stream": "0",
        "endpoint_set_id": "sas-hamster-saas2.rs-proxy",
        "endpoint_set_cluster": "sas"
    }
    """
    auto_backup = True
    sync_upload_to_mds = True
    ttl = 28
    releasable = True
    releasers = ["YABS_SERVER_SANDBOX_TESTS"]
    testenv_switch_trigger = sdk2.Attributes.String()


class YabsServerExtServiceEndpoint(BaseBackupSdk2Resource):
    """
    Endpoint set of external service

    Example:
    {
        "cluster": "sandbox",
        "endpoint_set": "service-tag-endpoint"
    }
    """
    auto_backup = True
    sync_upload_to_mds = True
    ttl = 28
    service_tag = sdk2.Attributes.String(required=True)


class YabsServerExtServiceStabilityReport(BaseBackupSdk2Resource):
    auto_backup = True
    sync_upload_to_mds = True


class GOALNET_RELEASE_TAR(YABS_RELEASE_BASE_RESOURCE):
    pass


class WORD_NET_RELEASE_TAR(YABS_RELEASE_BASE_RESOURCE):
    pass


class OFFER_MATCH_RELEASE_TAR(YABS_RELEASE_BASE_RESOURCE):
    pass


class YabsServerResponseDump(BaseBackupSdk2Resource):
    """ Resource with dolbilo dump """
    auto_backup = True
    restart_policy = ctr.RestartPolicy.DELETE
    sync_upload_to_mds = True
    ttl = 4

    pack_codec = sdk2.Attributes.String("UC codec used for dump compression")
    base_revision = sdk2.Attributes.Integer("Yabs-server base revision")
    ammo_type = sdk2.Attributes.String("Ammo type")
    debug_cookie = sdk2.Attributes.String("Debug cookie")
    debug_mode = sdk2.Attributes.String("Debug output mode")
    dump_parser_id = sdk2.Attributes.Integer("ID of response dump parser bound to this resource")
    content_type = sdk2.Attributes.String("Type of data contained in dump "
                                          "(most often cases are primary yabs-server response content and secondary count links responses")
    shoot_index = sdk2.Attributes.Integer("Index of reshoot attempt (0 for initial shooting session, 1 and more for actual reshoots)")
    shoot_reports_yt_node = sdk2.Attributes.String("Path to yt node where shoot reports are stored")
    yt_table_path = sdk2.Attributes.String("Path to raw YT table with results")


class YABS_SERVER_STAT_SPEC(BaseBackupSdk2Resource):
    """
    Spec for bs stat role.
    Contains resource ids of BS_RELEASE_TAR and YABS_SERVER_B2B_BINARY_BASE

    Format:
    {
        "bs_release_tar_resource_id": 2523948611,
        "bs_release_yt_resource_id": 2563522196,
        "binary_bases": {
            "st_update_245": 2523664900,
            ...
        }
    }
    """
    auto_backup = True
    sync_upload_to_mds = True
    ttl = 28

    revision = sdk2.Attributes.String("Arcadia revision")


class YABS_SERVER_AB_EXPERIMENT_SETTINGS(BaseBackupSdk2Resource):
    """Resource with json AB experiment settings"""

    auto_backup = True
    sync_upload_to_mds = True


class YabsServerPerformanceShootData(BaseBackupSdk2Resource):
    """Resource with performance task shoot data"""

    auto_backup = True
    sync_upload_to_mds = True
    server_role = sdk2.Attributes.String("meta/stat")


class YabsServerMetaStatProtobufImage(BaseBackupSdk2Resource):
    """
    ya buf build result of meta-stat protocol defined in:
      - yabs/server/libs/server/protocol/proto/stat_request.proto
      - yabs/server/libs/server/protocol/proto/pmatch_response.proto
    """
    releasable = True
    auto_backup = True
    sync_upload_to_mds = True
    releasers = ['YABS_SERVER_SANDBOX_TESTS']


class YabsServerMetaStatRequestFbs(BaseBackupSdk2Resource):
    """
    flatc -b --schema result for file:
      - yabs/server/libs/server/protocol/fbs/stat_request.fbs
    """
    releasable = True
    auto_backup = True
    releasers = ['YABS_SERVER_SANDBOX_TESTS']


class YabsServerMetaStatResponseFbs(BaseBackupSdk2Resource):
    """
    flatc -b --schema result for file:
      - yabs/server/libs/server/protocol/fbs/pmatch_response.fbs
    """
    releasable = True
    auto_backup = True
    releasers = ['YABS_SERVER_SANDBOX_TESTS']


class YABS_SERVER_GCDA_COVERAGE(sdk2.Resource):
    """.tar.gz archive with .gcda coverage files of yabs-server"""
    auto_backup = True


class YABS_SERVER_GCOV_COVERAGE(sdk2.Resource):
    """Directory with coverage files of yabs-server (.html or .gcov)"""
    auto_backup = True


class YABSCS_CI_PACKAGE_PATCHED(BaseBackupSdk2Resource):
    """
    Package for CS diff tests
    """
    sync_upload_to_mds = True
    ttl = 14


class YABSCS_CI_PACKAGE_TRUNK(BaseBackupSdk2Resource):
    """
    Package for CS diff tests
    """
    sync_upload_to_mds = True
    ttl = 14
