import json
import signal


JSONP_BEHAVIORS = (
    ("all_numeric_signs", "Numbers may be prefixed by any \'+\' and \'-\', e.g., +4, -+-+77", False),
    ("any_type_at_start", "A JSON document may start with any type, not just arrays or objects", False),
    ("comments", "JavaScript comments, both /*...*/ and //... styles", False),
    ("control_char_in_string", "Strings may contain raw control characters without \\u-escaping", False),
    ("hex_numbers", "Hexadecimal numbers, e.g., 0x1f", False),
    ("binary_numbers", "Binary numbers, e.g., 0b1001", False),
    ("octal_numbers", "New-style octal numbers, e.g., 0o731  (see leading-zeros for legacy octals)", False),
    ("initial_decimal_point", "Floating-point numbers may start with a decimal point (no units digit)", False),
    ("extended_unicode_escapes", "Extended Unicode escape sequence \\u{..} for non-BMP characters", False),
    ("js_string_escapes", "All JavaScript character \\-escape sequences may be in strings", True),
    ("leading_zeros", "Numbers may have extra leading zeros (see --leading-zero-radix option)", False),
    ("non_numbers", "Non-numbers may be used, such as NaN or Infinity", False),
    ("nonescape_characters", "Unknown character \\-escape sequences stand for that character (\\Q -> 'Q')", True),
    ("identifier_keys", "JavaScript identifiers are converted to strings when used as object keys", True),
    ("nonstring_keys", "Value types other than strings (or identifiers) may be used as object keys", False),
    ("omitted_array_elements", "Arrays may have omitted/elided elements, e.g., [1,,3] == [1,undefined,3]", False),
    ("single_quoted_strings", "Strings may be delimited with both double (\") and single (\') quotation marks", False),
    ("trailing_comma", "A final comma may end the list of array or object members", False),
    ("trailing_decimal_point", "Floating-point number may end with a decimal point and no following fractional digits", False),
    ("undefined_values", "The JavaScript 'undefined' value may be used", False),
    ("format_control_chars", "Unicode \"format control characters\" may appear in the input", False),
    ("unicode_whitespace", "Treat any Unicode whitespace character as valid whitespace", False),
    ("leading_zeros", "Numbers may have leading zeros", False),
    ("duplicate_keys", "Objects may have repeated keys", False),
    ("zero_byte", "Strings may contain U+0000, which may not be safe for C-based programs", False),
    ("bom", "A JSON document may start with a Unicode BOM (Byte Order Mark)", False),
    ("non_portable", "Anything technically valid but likely to cause data portability issues", True),
)


def try_parse_json_with_timeout(text, jsonp_parser=None, return_errors=True, timeout=2, **options):
    def raise_timeout(signum, frame):
        raise Exception("Parsing timed out")

    result = None
    error = None
    signal.signal(signal.SIGALRM, raise_timeout)
    signal.alarm(timeout)
    try:
        result = try_parse_json(text, jsonp_parser=jsonp_parser, return_errors=return_errors, **options)
    except Exception as exc:
        error = exc
    finally:
        signal.alarm(0)
    if error is not None:
        raise error
    return result


def try_parse_json(text, jsonp_parser=None, return_errors=True, **options):
    parsed = None
    errors = []
    if not jsonp_parser:
        import demjson
        jsonp_parser = demjson.decode

    try:
        parsed = json.loads(text)
        if return_errors:
            return parsed, []
        return parsed
    except ValueError:
        pass

    if return_errors:
        parsed, errors, _ = jsonp_parser(text, strict=True, return_errors=True, **options)
    else:
        parsed = jsonp_parser(text, strict=True, return_errors=False, **options)
    return (parsed, errors) if return_errors else parsed
