import logging
import os
import tarfile
from multiprocessing import cpu_count

from sandbox.projects.yabs.qa.utils.resource import sync_resource
from sandbox.common.fs import make_folder as mkdir
from sandbox.projects.common.yabs.server.util.general import run_process
from sandbox.projects.yabs.qa.resource_types import UCTool, YabsResponseDumpUnpacker

from sandbox.projects.yabs.sandbox_task_tracing import trace_calls
from sandbox.projects.yabs.sandbox_task_tracing.wrappers.sandbox.sdk2 import new_resource_data


DEFAULT_THREADS = cpu_count()
UNPACKER_EXECUTABLE = "dolbilo2json"


logger = logging.getLogger(__name__)


@trace_calls
def unpack_responses_dump_resource(task_instance, dump_resources, destination_path,
                                   uc_resource=None, unpacker_resource=None):
    tmp_dir = 'decompressed_dumps'
    mkdir(tmp_dir)
    unpacker = get_unpacker_executable_path(unpacker_resource)

    for resource in dump_resources:
        compressed_dump_path = sync_resource(resource)
        decompressed_dump_path = os.path.join(tmp_dir, str(resource.id))
        uc_decompress(task_instance, compressed_dump_path, decompressed_dump_path, uc_resource)
        unpack_responses(task_instance, decompressed_dump_path, destination_path, unpacker)


@trace_calls
def unpack_responses(task_instance, dump_path, destination_path, unpacker, threads=DEFAULT_THREADS):
    mkdir(destination_path)
    command = [
        unpacker,
        '--extract-proto',
        '-r', destination_path,
        '-j', str(threads),
        dump_path
    ]

    run_process(task_instance, command, logger_name='dolbilo2json:{}'.format(dump_path))


@trace_calls
def uc_decompress(task_instance, source_path, destination_path, uc_resource_id=None, threads=DEFAULT_THREADS):
    uc = get_uc(task_instance, uc_resource_id)

    command = uc + [
        '-d',
        '-C', 'lz4hc',
        '-f', source_path,
        '-t', destination_path,
        '-j', str(threads)
    ]

    run_process(task_instance, command, logger_name='uc')


@trace_calls
def uc_compress(task_instance, source_path, destination_path=None, pack_type='lz4hc', uc_resource_id=None, threads=DEFAULT_THREADS):
    uc = get_uc(task_instance, uc_resource_id)

    destination_path = destination_path or source_path + '.' + pack_type
    command = uc + [
        '-c',
        '-C', pack_type,
        '-f', source_path,
        '-t', destination_path,
        '-j', str(threads)
    ]
    run_process(task_instance, command, logger_name='uc')

    return destination_path


@trace_calls
def get_uc(task_instance, uc_resource=None):
    uc_executable_path = os.path.join('.', 'uc')

    if os.path.isfile(uc_executable_path):
        return [uc_executable_path]

    uc_tar_path = sync_resource(
        resource=uc_resource,
        resource_type=UCTool,
    )
    run_process(task_instance, ['tar', '-xzf', uc_tar_path, '-C', '.'], logger_name='tar')

    return [uc_executable_path]


def find_unpacker_resource(**attrs):
    return YabsResponseDumpUnpacker.find(attrs=attrs, limit=1).first()


def get_unpacker_executable_path(unpacker_resource):
    resource_data = new_resource_data(unpacker_resource)

    executable_path = os.path.join(str(resource_data.path), UNPACKER_EXECUTABLE)
    if tarfile.is_tarfile(str(resource_data.path)):
        extract_path = os.getcwd()
        with tarfile.open(str(resource_data.path), "r") as archive:
            archive.extractall(path=extract_path)
        executable_path = os.path.join(extract_path, UNPACKER_EXECUTABLE)

    return executable_path
