import logging
import socket

import requests
import six

logger = logging.getLogger(__name__)


class SaasKVRSCoordinatorClient():
    """Client for SaaS KVRS coordinator

    Coordinator API: https://a.yandex-team.ru/arcadia/search/plutonium/deploy/cpp_coordinator/coordinator.cpp
    """
    def __init__(self, cluster, endpoint_set_id):
        self._endpoint_set_id = endpoint_set_id
        self._cluster = cluster
        self._active_endpoint = None
        self._session = requests.Session()

    def _resolve_endpoints(self):
        """
        :rtype: list[infra.yp_service_discovery.api.api_pb2.TEndpoint]
        """
        from infra.yp_service_discovery.api import api_pb2
        from infra.yp_service_discovery.python.resolver.resolver import Resolver

        resolver = Resolver(client_name="saas_kvrs_cordinator_client:{}".format(socket.gethostname()), timeout=5)

        request = api_pb2.TReqResolveEndpoints()
        request.cluster_name = self._cluster
        request.endpoint_set_id = self._endpoint_set_id
        result = resolver.resolve_endpoints(request)

        return result.endpoint_set.endpoints

    def _get_active_endpoint(self):
        """
        :rtype: infra.yp_service_discovery.api.api_pb2.TEndpoint
        """
        if self._active_endpoint:
            return self._active_endpoint

        for endpoint in self._resolve_endpoints():
            try:
                url = six.moves.urllib.parse.urlunsplit([
                    "http", endpoint.fqdn, "", "", ""
                ])
                requests.get(url)
            except requests.exceptions.ConnectionError:
                logger.debug("Failed to connect to endpoint %s", url)
            else:
                self._active_endpoint = endpoint
                break

        return self._active_endpoint

    def _request(self, method, handler, _json=None):
        active_endpoint = self._get_active_endpoint()
        logger.debug("Active endpoint: %s", active_endpoint)
        url = six.moves.urllib.parse.urlunsplit([
            "http", active_endpoint.fqdn, handler, "", ""
        ])
        request = requests.Request(method.upper(), url, json=_json).prepare()
        request_string = "{method} {url} HTTP/1.1\r\n{headers}\r\n{body}".format(
            method=request.method,
            url=request.url,
            headers="\r\n".join("{}: {}".format(k, v) for k, v in request.headers.items()),
            body=request.body or "",
        )
        logger.debug("Request: %s", request_string)

        response = self._session.send(request)
        response.raise_for_status()
        return response.json()

    def list_frozen_states(self):
        """List frozen states. State is a dictionary in {"Namespace", "StateId"} format

        :return: List of frozen states
        :rtype: list[dict]
        """
        response = self._request("GET", "list_frozen_states")
        return response["states"]

    def freeze_last_state(self):
        """Freeze last state. State is a dictionary in {"Namespace", "StateId"} format

        :return: Frozen states
        :rtype: list[dict]
        """
        response = self._request("GET", "freeze_last_state")
        return response["states"]

    def unfreeze_state(self, states):
        """Unfreeze states. State is a dictionary in {"Namespace", "StateId"} format

        :param states: List of states to unfreeze
        :type states: list[dict]
        :return: Changed states
        :rtype: list[dict]
        """
        response = self._request("POST", "unfreeze_state", _json={"states": states})
        return response["states"]
