def topology_from_states(states):
    """Construct topology object from states.
    State format: {'Namespace': str, 'StateId': str}

    :param states: List of states
    :type states: List[Dict]
    :return: Topology object
    :rtype: search.plutonium.deploy.proto.rescan_pb2.TExtendedNotification
    """
    from search.plutonium.deploy.proto import rescan_pb2

    extended_notification = rescan_pb2.TExtendedNotification()
    for state in states:
        resource_notification = extended_notification.Resources.add()
        resource_notification.Namespace = "topology"
        resource_notification.Labels.Label["SnapshotId"] = state["StateId"]
        resource_notification.Labels.Label["Stream"] = state["Namespace"]

    return extended_notification


def topology_to_states(topology):
    """Construct state object from topology.
    State format: {'Namespace': str, 'StateId': str}

    :param topology: List of stream's description
    :type topology: List[TExtendedNotification]
    :return: List of dictionaries in {"Namespace", "StateId"} format
    :rtype: List[Dict(str, str)]
    """
    return [
        {
            "Namespace": item.Labels.Label["Stream"],
            "StateId": item.Labels.Label["SnapshotId"]
        }
        for item in topology.Resources
    ]


def topology_from_string(_str):
    """Construct topology object from string

    :param _str: Topology string
    :type _str: str
    :return: Topology object
    :rtype: search.plutonium.deploy.proto.rescan_pb2.TExtendedNotification
    """
    from google.protobuf.text_format import Parse
    from search.plutonium.deploy.proto import rescan_pb2

    extended_notification = rescan_pb2.TExtendedNotification()
    return Parse(_str, extended_notification)
