import logging

from sandbox import sdk2

from sandbox.common.errors import VaultError
from sandbox.projects.yabs.qa.solomon.push_client import SolomonPushClient

logger = logging.getLogger(__name__)


class SolomonTaskMixinParameters(sdk2.Parameters):
    push_to_solomon = sdk2.parameters.Bool("Push metrics to solomon", default=True)
    with push_to_solomon.value[True]:
        with sdk2.parameters.Group("Solomon parameters") as solomon_general_settings:
            solomon_hostname = sdk2.parameters.String("Solomon hostname", default='solomon.yandex.net')

        with sdk2.parameters.Group("Solomon sensor parameters") as solomon_sensor_settings:
            solomon_project = sdk2.parameters.String("Solomon project", default="yabs_testing")
            solomon_cluster = sdk2.parameters.String("Solomon cluster")
            solomon_service = sdk2.parameters.String("Solomon service")

        with sdk2.parameters.Group("Solomon auth parameters") as solomon_auth_settings:
            solomon_token_yav_secret_id = sdk2.parameters.String('Secret id with solomon OAuth token')
            solomon_token_yav_secret_version = sdk2.parameters.String('Version of secret')
            solomon_token_yav_secret_key = sdk2.parameters.String('Name of solomon Oauth token key in secret', default='solomon_token')

            solomon_token_vault_name = sdk2.parameters.String('Name of the vault with solomon token')

        with sdk2.parameters.Group("Solomon push settings") as solomon_push_settings:
            spack = sdk2.parameters.Bool("Use spack to push to solomon", default=False)
            max_sensors_per_push = sdk2.parameters.Integer('Maximum number of sensors to be pushed in a single request to Solomon', default=4000)


class SolomonTaskMixin(object):
    __solomon_token = None
    __solomon_push_client = None

    class Parameters(SolomonTaskMixinParameters):
        pass

    @property
    def solomon_token(self):
        if self.__solomon_token is None:
            self.__solomon_token = init_solomon_token(self)
        return self.__solomon_token

    @property
    def solomon_push_client(self):
        if self.__solomon_push_client is None:
            self.__solomon_push_client = init_push_client(self)
        return self.__solomon_push_client

    def on_success(self, prev_status):
        if self.Parameters.push_to_solomon:
            try:
                self.solomon_push_client.push_collected()
            except Exception as err:
                message = 'Failed to push metrics to solomon'
                logger.exception(message)
                self.set_info('{msg}: {err}'.format(msg=message, err=err))
        else:
            logger.debug('Debug mode is on, sensors are %s', self.solomon_push_client.sensors)

        return super(SolomonTaskMixin, self).on_success(prev_status)


def init_solomon_token(task):
    if task.Parameters.solomon_token_yav_secret_id:
        secret = sdk2.yav.Secret(task.Parameters.solomon_token_yav_secret_id, task.Parameters.solomon_token_yav_secret_version)
        return secret.data()[task.Parameters.solomon_token_yav_secret_key]

    if task.Parameters.solomon_token_vault_name:
        try:
            return sdk2.VaultItem(task.Parameters.solomon_token_vault_name).data()
        except VaultError as err:
            logging.warning('Failed to get data from vault: %s', err)

    return None


def init_push_client(task):
    logger.info("Initializing SolomonPushClient with parameters of %s", task.__class__.__name__)
    solomon_push_settings = {
        'max_sensors_per_push': task.Parameters.max_sensors_per_push,
        'spack': task.Parameters.spack,
    }
    return SolomonPushClient(
        task.Parameters.solomon_project,
        token=init_solomon_token(task),
        default_cluster=task.Parameters.solomon_cluster,
        default_service=task.Parameters.solomon_service,
        solomon_hostname=task.Parameters.solomon_hostname,
        **solomon_push_settings
    )
