import logging
from collections import defaultdict

import requests
from six.moves import urllib

from sandbox.projects.common.decorators import retries


logger = logging.getLogger(__name__)
SOLOMON_TIMEOUT_RESPONSE_CODE = 504


def chunk_split(iterable, chunk_size):
    for index in range(0, len(iterable), chunk_size):
        yield iterable[index:index + chunk_size]


class SolomonPushClient(object):
    def __init__(
        self, project,
        max_sensors_per_push=4000,
        default_cluster=None,
        default_service=None,
        token=None, spack=False,
        solomon_hostname='solomon.yandex.net',
    ):
        self.project = project
        self.default_cluster = default_cluster
        self.default_service = default_service

        self.max_sensors_per_push = max_sensors_per_push
        self.spack = spack

        self.solomon_hostname = solomon_hostname

        self.__token = token

        self._clear_sensors()

    def add(self, data, service=None, cluster=None, host=''):
        if service is None:
            service = self.default_service
        if cluster is None:
            cluster = self.default_cluster
        self.sensors[(cluster, service, host)].extend(data)

    def push_collected(self):
        logger.info('Send data to solomon')
        for (cluster, service, host), sensors in self.sensors.iteritems():
            self.__do_push_sensors(cluster, service, host, sensors)
            logger.debug('Sent to solomon: service: {}; data: {}'.format(service, sensors))
        self._clear_sensors()
        logger.info('Sent to solomon successfully')

    def _clear_sensors(self):
        self.sensors = defaultdict(list)

    @retries(max_tries=3, delay=10, exceptions=(requests.exceptions.HTTPError, ))
    def __do_post(self, solomon_push_url, spack_data, json_data, headers):
        r = requests.post(
            solomon_push_url,
            json=json_data,
            data=spack_data,
            headers=headers,
        )
        if r.status_code not in (requests.codes.ok, requests.codes.accepted):
            logger.error("Got %s %s", r.status_code, r.content)
        r.raise_for_status()
        return r.content

    def __do_push_sensors(self, cluster, service, host, sensors_list):
        if self.spack:
            try:
                from yabs.server.libs.py_json2spack import encode_spackv1
            except ImportError:
                logger.warning('Cannot import py_json2spack, falling back to json')
                self.spack = False

        ack = 'at_least_one'
        params = dict(project=self.project, service=service, cluster=cluster)
        if self.__token:
            scheme, api_url = 'https', 'api/v2/push'
        else:
            scheme, api_url = 'http', 'push'
            params['ack'] = ack

        solomon_push_url = urllib.parse.urlunparse(
            urllib.parse.ParseResult(
                scheme=scheme,
                netloc=self.solomon_hostname,
                path=api_url,
                params='',
                query=urllib.parse.urlencode(params),
                fragment='',
            )
        )

        for index, sensors_chunk in enumerate(chunk_split(sensors_list, self.max_sensors_per_push)):
            logger.info('Chunk #{}'.format(index))
            data = {
                'commonLabels': {
                    'host': host,
                },
                'sensors': sensors_chunk,
            }
            logger.debug('Sending data to {}: {}'.format(solomon_push_url, data))
            headers = {}
            spack_data = None
            json_data = None
            if self.spack:
                spack_data = encode_spackv1(data)
                headers['Content-Type'] = 'application/x-solomon-spack'
            else:
                json_data = data
            if self.__token:
                headers['Authorization'] = 'OAuth {}'.format(self.__token)

            self.__do_post(solomon_push_url, spack_data, json_data, headers)
