import json
import logging
import six

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types.resource import State
from sandbox.common.types.task import Status
from sandbox.projects.common.yabs.server.util.general import check_tasks
from sandbox.projects.yabs.qa.utils.general import get_task_html_hyperlink
from sandbox.projects.yabs.qa.spec.constants import META_MODES

logger = logging.getLogger(__name__)


def generate_shard_key(shard, meta_mode=None):
    shard_key = 'tags_shard_{}'.format(shard)
    if meta_mode is not None:
        shard_key += '_' + meta_mode
    return shard_key


def iter_tags(shard_nums, meta_modes=META_MODES):
    for meta_mode in meta_modes:
        for shard in shard_nums:
            yield generate_shard_key(shard, meta_mode=meta_mode)
        yield 'tags_common_{}'.format(meta_mode)


def iter_meta_base_tags_fields(shard_keys=(), meta_modes=META_MODES):
    for meta_mode in meta_modes:
        yield 'base_tags_meta_{}'.format(meta_mode)


def iter_stat_base_tags_fields(shard_keys, meta_modes=META_MODES):
    for meta_mode in meta_modes:
        for shard_key in shard_keys:
            yield 'base_tags_stat_{}_{}'.format(meta_mode, shard_key)
        yield 'base_tags_stat_{}_COMMON'.format(meta_mode)


def iter_base_tags_fields(shard_keys, meta_modes=META_MODES):
    for meta_mode in meta_modes:
        for shard_key in shard_keys:
            yield 'base_tags_stat_{}_{}'.format(meta_mode, shard_key)
        yield 'base_tags_stat_{}_COMMON'.format(meta_mode)
        yield 'base_tags_meta_{}'.format(meta_mode)


def get_stat_base_tags_by_meta_mode(base_tags_map, shard_keys, meta_modes=META_MODES):
    base_tags_by_meta_mode = {}
    for meta_mode in meta_modes:
        for shard_key in shard_keys:
            mbb_key = 'base_tags_stat_{}'.format(meta_mode)
            base_tags_by_meta_mode[mbb_key] = list(
                set(base_tags_by_meta_mode.get(mbb_key, []) + base_tags_map['base_tags_stat_{}_{}'.format(meta_mode, shard_key)])
            )
    return base_tags_by_meta_mode


def get_meta_base_tags_by_meta_mode(base_tags_map, meta_modes=META_MODES):
    base_tags_by_meta_mode = {}
    for meta_mode in meta_modes:
        base_tags_by_meta_mode['base_tags_meta_{}'.format(meta_mode)] = list(set(base_tags_map['base_tags_meta_{}'.format(meta_mode)]))
    return base_tags_by_meta_mode


def get_base_tags_by_meta_mode(base_tags_map, shard_keys, meta_modes=META_MODES):
    base_tags_by_meta_mode = {}
    for meta_mode in meta_modes:
        base_tags_by_meta_mode['base_tags_meta_{}'.format(meta_mode)] = list(set(base_tags_map['base_tags_meta_{}'.format(meta_mode)]))
        for shard_key in shard_keys:
            mbb_key = 'base_tags_stat_{}'.format(meta_mode)
            base_tags_by_meta_mode[mbb_key] = list(
                set(base_tags_by_meta_mode.get(mbb_key, []) + base_tags_map['base_tags_stat_{}_{}'.format(meta_mode, shard_key)])
            )
    return base_tags_by_meta_mode


def filter_base_tags_by_meta_mode(base_tags_by_meta_mode, needed_bases):
    filtered_base_tags_by_meta_mode = {}
    for key, base_tags in base_tags_by_meta_mode.items():
        filtered_base_tags_by_meta_mode[key] = list(set(base_tags) & set(needed_bases))
    return filtered_base_tags_by_meta_mode


def get_binary_base_resource_ids(binary_base_tags_map, binary_base_resource_id_by_tag, shard_nums, meta_mode):
    binary_base_resource_ids = []
    for shard_key in iter_tags(shard_nums, meta_modes=[meta_mode]):
        for base_tag in binary_base_tags_map[shard_key]:
            binary_base_resource_ids.append(binary_base_resource_id_by_tag[base_tag])
    return binary_base_resource_ids


def get_base_resource_ids(binary_base_tags_map, binary_base_resource_id_by_tag, shard_keys, meta_mode, server_mode='meta'):
    binary_base_resource_ids = set()
    if server_mode == 'meta':
        for shard_key_field in iter_meta_base_tags_fields(meta_modes=[meta_mode]):
            logging.debug('Getting bases from %s', shard_key_field)
            for base_tag in binary_base_tags_map[shard_key_field]:
                binary_base_resource_ids.add(binary_base_resource_id_by_tag[base_tag])
    elif server_mode == 'stat':
        for shard_key_field in iter_stat_base_tags_fields(shard_keys, meta_modes=[meta_mode]):
            logging.debug('Getting bases from %s', shard_key_field)
            for base_tag in binary_base_tags_map[shard_key_field]:
                binary_base_resource_ids.add(binary_base_resource_id_by_tag[base_tag])
    return list(binary_base_resource_ids)


def create_spec_resource(task, resource_class, spec, resource_description, resource_file_name, **attrs):
    spec_resource = resource_class(task, resource_description, resource_file_name, **attrs)
    spec_resource_data = sdk2.ResourceData(spec_resource)
    with open(str(spec_resource_data.path), 'w') as spec_file:
        json.dump(spec, spec_file, indent=2, sort_keys=True)
    spec_resource_data.ready()
    return spec_resource


def filter_not_ok_tasks(task, task_ids):
    task_statuses = check_tasks(task, task_ids, raise_on_fail=False)
    broken_tasks = [
        task_obj.id
        for task_obj, task_status in task_statuses
        if task_status in Status.Group.BREAK
    ]
    failed_tasks = [
        task_obj.id
        for task_obj, task_status in task_statuses
        if task_status not in Status.Group.SUCCEED | Status.Group.BREAK
    ]
    return broken_tasks, failed_tasks


def fail_if_subtasks_not_ok(task, broken_subtasks, failed_subtasks):
    if failed_subtasks or broken_subtasks:
        task.set_info(
            "Spec generation was not successful due to failure of {}".format(
                ", ".join([get_task_html_hyperlink(task_id) for task_id in failed_subtasks + broken_subtasks])
            ),
            do_escape=False,
        )
        if failed_subtasks:
            raise TaskFailure('Spec generation failed. This task can not be restarted.')
        else:
            raise Exception('Spec generation failed. Subtasks and this task may be restarted manually.')


def find_last_released_resource(res_type, attrs, **kwargs):
    if isinstance(res_type, six.string_types):
        res_type = sdk2.Resource[res_type]
    resources = res_type.find(attrs=attrs, state=State.READY, order="-id", **kwargs).limit(20)
    last_resource = None
    for resource in resources:
        logger.debug('Checking resource #%s %s', resource.id, dir(resource))
        if not last_resource or (hasattr(resource, 'major_version') and int(last_resource.major_version) < int(resource.major_version)):
            last_resource = resource
    return last_resource
