import itertools
import json
import logging
from itertools import chain

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types.task import ReleaseStatus
from sandbox.common.types.misc import NotExists
from sandbox.projects.yabs.qa.pipeline.stage import stage
from sandbox.projects.common.yabs.server.util.general import find_last_ready_resource, check_tasks
from sandbox.projects.yabs.qa.resource_types import (
    BS_RELEASE_TAR,
    BS_RELEASE_YT,
    YABS_CS_INPUT_SPEC,
    YABS_CS_SETTINGS_ARCHIVE,
    YABS_MYSQL_ARCHIVE_CONTENTS,
    YABS_SERVER_DOLBILKA_PLAN,
    YABS_SERVER_REQUEST_LOG_GZ,
    YABS_SERVER_TESTENV_SHARD_MAP,
    YABS_SERVER_TESTENV_DB_FLAGS,
    YabsServerExtServiceEndpoint,
)
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShoot2 import YabsServerB2BFuncShoot2
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootStability import YabsServerB2BFuncShootStability
from sandbox.projects.yabs.qa.tasks.YabsServerValidateResponses import YabsServerValidateResponses
from sandbox.projects.yabs.qa.tasks.YabsServerStatPerformancePrepareDplan import YabsServerStatPerformancePrepareDplan
from sandbox.projects.yabs.qa.tasks.YabsServerPrepareStatStub import YabsServerPrepareStatStub
from sandbox.projects.yabs.qa.utils import task_run_type

from sandbox.projects.yabs.qa.spec.constants import META_MODES, CLUSTER_BY_META_MODE, SPLIT_HANDLERS_BY_META_MODE
from sandbox.projects.yabs.qa.spec.misc import get_base_resource_ids, find_last_released_resource, iter_stat_base_tags_fields, iter_meta_base_tags_fields
from sandbox.projects.YabsLmService.resource_types import YABS_LINEAR_MODELS_EXECUTABLE

# https://st.yandex-team.ru/SHMDUTY-160
FT_ENGINE_ENV = {
    "Y_NO_AVX_IN_DOT_PRODUCT": "1"
}


@stage(provides=(
    "stat_bs_release_tar_resource_id",
    "stat_bs_release_yt_resource_id",
    "meta_bs_release_tar_resource_id",
    "meta_bs_release_yt_resource_id",
), result_is_dict=True)
def get_engine_resources(task):
    stat_bs_release_tar_resource = task.Parameters.stat_bs_release_tar_resource or find_last_released_resource(
        BS_RELEASE_TAR, {"released": ReleaseStatus.STABLE, "build_mode": "release", "component_name": "yabs_server"},
    )
    stat_bs_release_yt_resource = task.Parameters.stat_bs_release_yt_resource or find_last_ready_resource(BS_RELEASE_YT, {"build_mode": "release"}, task_id=stat_bs_release_tar_resource.task.id)

    if task.Parameters.meta_bs_release_tar_resource:
        meta_bs_release_tar_resource = task.Parameters.meta_bs_release_tar_resource
    else:
        _meta_bs_release_tar_resource = find_last_released_resource(
            BS_RELEASE_TAR, {"released": ReleaseStatus.STABLE, "build_mode": "release", "component_name": "yabs_server_meta"},
        )
        if not _meta_bs_release_tar_resource:
            meta_bs_release_tar_resource = stat_bs_release_tar_resource
        else:
            meta_bs_release_tar_resource = max(stat_bs_release_tar_resource, _meta_bs_release_tar_resource, key=lambda res: res.id)

    meta_bs_release_yt_resource = task.Parameters.meta_bs_release_yt_resource or find_last_ready_resource(BS_RELEASE_YT, {"build_mode": "release"}, task_id=meta_bs_release_tar_resource.task.id)

    return {
        "stat_bs_release_tar_resource_id": stat_bs_release_tar_resource.id,
        "stat_bs_release_yt_resource_id": stat_bs_release_yt_resource.id,
        "meta_bs_release_tar_resource_id": meta_bs_release_tar_resource.id,
        "meta_bs_release_yt_resource_id": meta_bs_release_yt_resource.id,
    }


@stage(provides=(
    "mysql_archive_resource_id",
    "cs_input_spec_resource_id",
    "cs_settings_archive_resource_id",
), result_is_dict=True)
def get_data_resources(task):
    cs_input_spec_resource = task.Parameters.cs_input_spec_resource or find_last_ready_resource(YABS_CS_INPUT_SPEC, {'testenv_switch_trigger': None})
    mysql_archive_resource = task.Parameters.mysql_archive_resource or find_last_ready_resource(YABS_MYSQL_ARCHIVE_CONTENTS, {'testenv_switch_trigger': None})
    cs_settings_archive_resource = task.Parameters.cs_settings_archive_resource or find_last_ready_resource(YABS_CS_SETTINGS_ARCHIVE, {'testenv_switch_trigger': None})

    return {
        "mysql_archive_resource_id": mysql_archive_resource.id,
        "cs_input_spec_resource_id": cs_input_spec_resource.id,
        "cs_settings_archive_resource_id": cs_settings_archive_resource.id,
    }


@stage(provides=(
    "ft_request_log_resource_id_map",
    "stat_load_request_log_resource_id_map",
    "meta_load_request_log_resource_id_map",
    "dolbilka_plan_resource_id_map",
), result_is_dict=True)
def get_ammo_resources(task):
    def get_or_find_resource_id(meta_mode, mapping_from_parameters, resource_type, test_type):
        resource_id_from_parameters = mapping_from_parameters.get(meta_mode)
        if resource_id_from_parameters:
            return resource_id_from_parameters
        return find_last_ready_resource(resource_type, {'testenv_switch_trigger_{}_{}'.format(meta_mode, test_type): None}).id

    dolbilka_plan_resource_id_map = {
        meta_mode: get_or_find_resource_id(meta_mode, task.Parameters.dolbilka_plan_resource_id_map, YABS_SERVER_DOLBILKA_PLAN, 'load')
        for meta_mode in META_MODES
    }
    ft_request_log_resource_id_map = {
        meta_mode: get_or_find_resource_id(meta_mode, task.Parameters.ft_request_log_resource_id_map, YABS_SERVER_REQUEST_LOG_GZ, 'func')
        for meta_mode in META_MODES
    }
    stat_load_request_log_resource_id_map = {
        meta_mode: get_or_find_resource_id(meta_mode, task.Parameters.stat_load_request_log_resource_id_map, YABS_SERVER_REQUEST_LOG_GZ, 'load')
        for meta_mode in META_MODES
    }
    meta_load_request_log_resource_id_map = {
        meta_mode: get_or_find_resource_id(meta_mode, task.Parameters.meta_load_request_log_resource_id_map, YABS_SERVER_REQUEST_LOG_GZ, 'meta_load')
        for meta_mode in META_MODES
    }

    return {
        "dolbilka_plan_resource_id_map": dolbilka_plan_resource_id_map,
        "ft_request_log_resource_id_map": ft_request_log_resource_id_map,
        "stat_load_request_log_resource_id_map": stat_load_request_log_resource_id_map,
        "meta_load_request_log_resource_id_map": meta_load_request_log_resource_id_map,
    }


@stage(provides=(
    "shard_map_resource_id",
    "gen_bin_bases_flags_resource_id",
    "ft_shoot_settings_resource_id",
    "load_shoot_settings_resource_id",
    "meta_load_shoot_settings_resource_id",
    "linear_models_binary_resource_id",
    "ext_service_endpoint_resource_ids",
), result_is_dict=True)
def get_misc_resources(task, hamster_ext_service_tags=None):
    hamster_ext_service_tags = hamster_ext_service_tags or {}

    shard_map_resource = task.Parameters.shard_map_resource or find_last_ready_resource(YABS_SERVER_TESTENV_SHARD_MAP, {'testenv_switch_trigger': None})
    gen_bin_bases_flags_resource = task.Parameters.gen_bin_bases_flags_resource or find_last_ready_resource(YABS_SERVER_TESTENV_DB_FLAGS, {'testenv_switch_trigger_base_gen': None})
    ft_shoot_settings_resource = task.Parameters.ft_shoot_settings_resource or find_last_ready_resource(YABS_SERVER_TESTENV_DB_FLAGS, {"test_type": 'ft_shoot'})
    load_shoot_settings_resource = task.Parameters.load_shoot_settings_resource or find_last_ready_resource(YABS_SERVER_TESTENV_DB_FLAGS, {"test_type": 'load_shoot'})
    meta_load_shoot_settings_resource = task.Parameters.meta_load_shoot_settings_resource or find_last_ready_resource(YABS_SERVER_TESTENV_DB_FLAGS, {"test_type": 'meta_load_shoot'})
    linear_models_binary_resource = task.Parameters.linear_models_binary_resource or find_last_ready_resource(YABS_LINEAR_MODELS_EXECUTABLE, {"released": ReleaseStatus.STABLE})

    ext_service_endpoint_resources = task.Parameters.ext_service_endpoint_resources or []
    explicitly_set_ext_service_endpoint_resources_tags = set([resource.service_tag for resource in ext_service_endpoint_resources])
    for hamster_ext_service_tag in set(itertools.chain.from_iterable(hamster_ext_service_tags.values())) - explicitly_set_ext_service_endpoint_resources_tags:
        ext_service_endpoint_resources.append(find_last_ready_resource(
            YabsServerExtServiceEndpoint,
            {'service_tag': hamster_ext_service_tag, 'testenv_switch_trigger': None}
        ))

    return {
        "shard_map_resource_id": shard_map_resource.id,
        "gen_bin_bases_flags_resource_id": gen_bin_bases_flags_resource.id,
        "ft_shoot_settings_resource_id": ft_shoot_settings_resource.id,
        "load_shoot_settings_resource_id": load_shoot_settings_resource.id,
        "meta_load_shoot_settings_resource_id": meta_load_shoot_settings_resource.id,
        "linear_models_binary_resource_id": linear_models_binary_resource.id,
        "ext_service_endpoint_resource_ids": [resource.id for resource in ext_service_endpoint_resources],
    }


def get_shards(task, shard_map):
    ft_shards = [shard_num for shard_name, shard_num in shard_map.items() if shard_name in task.Parameters.ft_shards]
    stat_load_shards = [shard_num for shard_name, shard_num in shard_map.items() if shard_name in task.Parameters.stat_load_shards] if task.Parameters.stat_load else []
    meta_load_shards = [shard_num for shard_name, shard_num in shard_map.items() if shard_name in task.Parameters.meta_load_shards] if task.Parameters.meta_load else []
    return ft_shards, stat_load_shards, stat_load_shards, meta_load_shards


def _get_shard_map(shard_map_resource_id):
    shard_map_resource = sdk2.Resource[shard_map_resource_id]
    with open(str(sdk2.ResourceData(shard_map_resource).path), 'r') as shard_map_file:
        shard_map = json.load(shard_map_file)
    return shard_map


@stage(provides='shard_map')
def get_shard_map(task, shard_map_resource_id):
    return _get_shard_map(shard_map_resource_id)


@stage(provides=('ft_shard_nums', 'load_shard_nums', 'stat_load_shard_nums', 'meta_load_shard_nums'))
def get_shard_nums(task, shard_map_resource_id):
    return get_shards(task, _get_shard_map(shard_map_resource_id))


@stage(provides=('ft_shard_keys', 'load_shard_keys', 'stat_load_shard_keys', 'meta_load_shard_keys'))
def get_shard_keys(task, shard_map_resource_id):
    ft_shard_keys = task.Parameters.ft_shards
    stat_load_shard_keys = task.Parameters.stat_load_shards if task.Parameters.stat_load else []
    meta_load_shard_keys = task.Parameters.meta_load_shards if task.Parameters.meta_load else []
    return ft_shard_keys, stat_load_shard_keys, stat_load_shard_keys, meta_load_shard_keys


@stage(provides=('stat_setup_ya_make_task_id', 'meta_setup_ya_make_task_id'))
def launch_setup_ya_make(
        task,
        shard_map_resource_id,
        stat_bs_release_yt_resource_id,
        stat_bs_release_tar_resource_id,
        meta_bs_release_yt_resource_id,
        meta_bs_release_tar_resource_id,
):
    def _launch_setup_ya_make_task(bs_release_yt_resource_id, bs_release_tar_resource_id, server_desc=''):
        return sdk2.Task['YABS_SERVER_SETUP_YA_MAKE'](
            task,
            description='Setup for {} BS_RELEASE_TAR #{}, {}'.format(server_desc, bs_release_tar_resource_id, task.Parameters.description),
            tags=task.Parameters.tags,
            server_resource=bs_release_tar_resource_id,
            cs_resource=bs_release_yt_resource_id,
            shard_map_res_id=shard_map_resource_id,
        ).enqueue()
    return (
        _launch_setup_ya_make_task(stat_bs_release_yt_resource_id, stat_bs_release_tar_resource_id, server_desc='stat').id,
        _launch_setup_ya_make_task(meta_bs_release_yt_resource_id, meta_bs_release_tar_resource_id, server_desc='meta').id,
    )


def get_base_tags_map(setup_ya_make_task_id, shard_keys, iter_base_tags_fields_iterator):
    setup_ctx = sdk2.Task[setup_ya_make_task_id].Context
    base_tags_map = {}
    for shard_tags_key in iter_base_tags_fields_iterator(shard_keys):
        shard_tags = getattr(setup_ctx, shard_tags_key)
        if shard_tags is NotExists:
            raise TaskFailure("Task {} has no '{}' context field, check your shard_map parameter".format(setup_ya_make_task_id, shard_tags_key))
        base_tags_map[shard_tags_key] = shard_tags
    return setup_ctx.db_ver, setup_ctx.cs_import_ver, base_tags_map


@stage(provides=(
    'stat_base_tags_map',
    'meta_base_tags_map',
    'stat_base_ver',
    'meta_base_ver',
    'stat_cs_import_ver',
    'meta_cs_import_ver',
))
def get_binary_base_tags(
        task,
        stat_setup_ya_make_task_id,
        meta_setup_ya_make_task_id,
        ft_shard_keys,
        stat_load_shard_keys,
        meta_load_shard_keys,
):
    check_tasks(task, [stat_setup_ya_make_task_id, meta_setup_ya_make_task_id])
    shard_keys = set(ft_shard_keys + stat_load_shard_keys + meta_load_shard_keys)

    stat_base_ver, stat_cs_import_ver, stat_base_tags_map = get_base_tags_map(stat_setup_ya_make_task_id, shard_keys, iter_stat_base_tags_fields)
    meta_base_ver, meta_cs_import_ver, meta_base_tags_map = get_base_tags_map(meta_setup_ya_make_task_id, shard_keys, iter_meta_base_tags_fields)

    return stat_base_tags_map, meta_base_tags_map, stat_base_ver, meta_base_ver, stat_cs_import_ver, meta_cs_import_ver


@stage(provides='ft_shoot_tasks')
def launch_ft_shoot_tasks(
        task,
        stat_bs_release_tar_resource_id,
        meta_bs_release_tar_resource_id,
        linear_models_binary_resource_id,
        linear_models_data_resource_id,
        ft_request_log_resource_id_map,
        stat_base_tags_map,
        stat_binary_base_resource_id_by_tag,
        meta_base_tags_map,
        meta_binary_base_resource_id_by_tag,
        ft_shard_nums,
        ft_shard_keys,
        shard_map_resource_id,
        ft_shoot_settings_resource_id,
        ext_service_endpoint_resource_ids,
        use_default_debug_cookie=True,
        shoot_threads=20,
        headers_update_dict=None,
        response_dumps_ttl=7,
        hamster_ext_service_tags=None,
):
    ft_shoot_tasks = {
        meta_mode: {}
        for meta_mode in CLUSTER_BY_META_MODE.keys()
    }
    shard_map = _get_shard_map(shard_map_resource_id)
    hamster_ext_service_tags = hamster_ext_service_tags or {}

    for meta_mode, cluster in CLUSTER_BY_META_MODE.items():
        request_log_resource_id = ft_request_log_resource_id_map[meta_mode]
        request_log_resource = sdk2.Resource[request_log_resource_id]
        cachedaemon_dump_resource_id = request_log_resource.cachedaemon_dump_res_id

        for ft_shard_key in ft_shard_keys:
            ft_shard_num = shard_map[ft_shard_key]

            stat_binary_base_resource_ids = get_base_resource_ids(stat_base_tags_map, stat_binary_base_resource_id_by_tag, [ft_shard_key], cluster, server_mode='stat')
            meta_binary_base_resource_ids = get_base_resource_ids(meta_base_tags_map, meta_binary_base_resource_id_by_tag, [ft_shard_key], cluster, server_mode='meta')

            binary_base_resource_ids = list(set(stat_binary_base_resource_ids) | set(meta_binary_base_resource_ids))

            task_parameters = dict(
                description='shoot {}, {}'.format(meta_mode, task.Parameters.description),
                tags=task.Parameters.tags,
                __requirements__={'tasks_resource': task.Requirements.tasks_resource},
                stat_server_resource=stat_bs_release_tar_resource_id,
                meta_server_resource=meta_bs_release_tar_resource_id,
                meta_mode=meta_mode,
                specify_cluster_set_config=True,
                stat_shards=[int(ft_shard_num)],
                binary_base_resources=binary_base_resource_ids,
                stat_binary_base_resources=stat_binary_base_resource_ids,
                meta_binary_base_resources=meta_binary_base_resource_ids,
                cache_daemon_stub_resource=cachedaemon_dump_resource_id,
                requestlog_resource=request_log_resource_id,
                update_parameters_resource=ft_shoot_settings_resource_id,
                update_parameters_resources=[ft_shoot_settings_resource_id],
                use_default_debug_cookie=use_default_debug_cookie,
                shoot_threads=shoot_threads,
                use_separate_linear_models_service=True,
                linear_models_binary_resource=linear_models_binary_resource_id,
                linear_models_data_resource=linear_models_data_resource_id,
                response_dumps_ttl=response_dumps_ttl,
                ext_service_endpoint_resources=ext_service_endpoint_resource_ids,
                hamster_ext_service_tags=hamster_ext_service_tags.get(meta_mode, []),
                meta_custom_environment=FT_ENGINE_ENV,
                stat_custom_environment=FT_ENGINE_ENV,
                use_base_state_generator=False,
                allow_reuse_of_this_task=True,
            )
            if headers_update_dict:
                task_parameters.update(headers_update_dict=headers_update_dict)

            ft_shoot_tasks[meta_mode][str(ft_shard_num)] = YabsServerB2BFuncShoot2(
                task,
                **task_parameters
            ).enqueue().id

    return ft_shoot_tasks


@stage(provides='ft_stability_shoot_tasks')
def launch_ft_stability_shoot_tasks(
        task,
        stat_bs_release_tar_resource_id,
        meta_bs_release_tar_resource_id,
        linear_models_binary_resource_id,
        linear_models_data_resource_id,
        ft_request_log_resource_id_map,
        stat_base_tags_map,
        stat_binary_base_resource_id_by_tag,
        meta_base_tags_map,
        meta_binary_base_resource_id_by_tag,
        ft_shard_nums,
        ft_shard_keys,
        shard_map_resource_id,
        ft_shoot_settings_resource_id,
        ext_service_endpoint_resource_ids,
        use_default_debug_cookie=True,
        stability_runs=4,
        shoot_threads=20,
        headers_update_dict=None,
        hamster_ext_service_tags=None,
):
    ft_stability_shoot_tasks = {
        meta_mode: {}
        for meta_mode in CLUSTER_BY_META_MODE.keys()
    }
    shard_map = _get_shard_map(shard_map_resource_id)
    hamster_ext_service_tags = hamster_ext_service_tags or {}

    for meta_mode, cluster in CLUSTER_BY_META_MODE.items():
        request_log_resource_id = ft_request_log_resource_id_map[meta_mode]
        request_log_resource = sdk2.Resource[request_log_resource_id]
        cachedaemon_dump_resource_id = request_log_resource.cachedaemon_dump_res_id

        for ft_shard_key in ft_shard_keys:
            ft_shard_num = shard_map[ft_shard_key]

            stat_binary_base_resource_ids = get_base_resource_ids(stat_base_tags_map, stat_binary_base_resource_id_by_tag, [ft_shard_key], cluster, server_mode='stat')
            meta_binary_base_resource_ids = get_base_resource_ids(meta_base_tags_map, meta_binary_base_resource_id_by_tag, [ft_shard_key], cluster, server_mode='meta')

            binary_base_resource_ids = list(set(stat_binary_base_resource_ids) | set(meta_binary_base_resource_ids))

            ft_stability_shoot_tasks[meta_mode][str(ft_shard_num)] = YabsServerB2BFuncShootStability(
                task,
                description='stability shoot {} | {}'.format(meta_mode, task.Parameters.description),
                tags=task.Parameters.tags,
                __requirements__={'tasks_resource': task.Requirements.tasks_resource},
                upload_to_yt_prefix='',
                stat_server_resource=stat_bs_release_tar_resource_id,
                meta_server_resource=meta_bs_release_tar_resource_id,
                meta_mode=meta_mode,
                specify_cluster_set_config=True,
                stat_shards=[int(ft_shard_num)],
                binary_base_resources=binary_base_resource_ids,
                stat_binary_base_resources=stat_binary_base_resource_ids,
                meta_binary_base_resources=meta_binary_base_resource_ids,
                cache_daemon_stub_resource=cachedaemon_dump_resource_id,
                requestlog_resource=request_log_resource_id,
                update_parameters_resource=ft_shoot_settings_resource_id,
                update_parameters_resources=[ft_shoot_settings_resource_id],
                use_default_debug_cookie=use_default_debug_cookie,
                shoot_threads=shoot_threads,
                stability_runs=stability_runs,
                headers_update_dict=headers_update_dict or {},
                use_separate_linear_models_service=True,
                linear_models_binary_resource=linear_models_binary_resource_id,
                linear_models_data_resource=linear_models_data_resource_id,
                ext_service_endpoint_resources=ext_service_endpoint_resource_ids,
                hamster_ext_service_tags=hamster_ext_service_tags.get(meta_mode, []),
                meta_custom_environment=FT_ENGINE_ENV,
                stat_custom_environment=FT_ENGINE_ENV,
                use_base_state_generator=False,
            ).enqueue().id

    return ft_stability_shoot_tasks


@stage(provides='ft_validation_tasks')
def launch_ft_validation(task, ft_shoot_tasks):
    check_tasks(
        task,
        list(chain(*(shard_tasks.values() for shard_tasks in ft_shoot_tasks.values()))),
    )
    validation_tasks = {}
    for meta_mode, shard_ft_shoot_tasks in ft_shoot_tasks.items():
        for shard_num, ft_shoot_task_id in shard_ft_shoot_tasks.items():
            validation_tasks.setdefault(meta_mode, {})[shard_num] = YabsServerValidateResponses(
                task,
                tags=task.Parameters.tags,
                description='Validate responses for {}, {}'.format(meta_mode, task.Parameters.description),
                shoot_task=ft_shoot_task_id,
                __requirements__={'tasks_resource': task.Requirements.tasks_resource},
            ).enqueue().id
    return validation_tasks


@stage(provides='stat_load_baseline_tasks')
def launch_stat_load_baseline(
        task,
        stat_bs_release_tar_resource_id,
        meta_bs_release_tar_resource_id,
        stat_load_request_log_resource_id_map,
        stat_base_tags_map,
        stat_binary_base_resource_id_by_tag,
        meta_base_tags_map,
        meta_binary_base_resource_id_by_tag,
        load_shard_nums,
        load_shard_keys,
        load_shoot_settings_resource_id,
        shard_map_resource_id,
):
    stat_load_baseline_tasks = {
        meta_mode: {}
        for meta_mode in CLUSTER_BY_META_MODE.keys()
    }

    shard_map = _get_shard_map(shard_map_resource_id)

    # Check if stat load is enabled inside the function
    # to always fill "stat_load_baseline_tasks" context field.
    # Otherwise "get_spec_data" stage will fail
    # because of absence of "stat_load_baseline_tasks" in context
    if task.Parameters.stat_load:
        for meta_mode, cluster in CLUSTER_BY_META_MODE.items():
            request_log_resource_id = stat_load_request_log_resource_id_map[meta_mode]
            request_log_resource = sdk2.Resource[request_log_resource_id]
            cachedaemon_dump_id = request_log_resource.cachedaemon_dump_res_id

            for shard_key in load_shard_keys:
                shard_num = shard_map[shard_key]
                logging.debug('Scheduling stat_load for %s %s %s', meta_mode, shard_num, shard_key)

                stat_binary_base_resource_ids = get_base_resource_ids(stat_base_tags_map, stat_binary_base_resource_id_by_tag, [shard_key], cluster, server_mode='stat')
                meta_binary_base_resource_ids = get_base_resource_ids(meta_base_tags_map, meta_binary_base_resource_id_by_tag, [shard_key], cluster, server_mode='meta')

                stat_load_baseline_tasks[meta_mode][str(shard_num)] = YabsServerStatPerformancePrepareDplan(
                    task,
                    description='shoot {}, shard {}, {}'.format(meta_mode, shard_num, task.Parameters.description),
                    tags=task.Parameters.tags,
                    stat_binary_base_resources=stat_binary_base_resource_ids,
                    meta_binary_base_resources=meta_binary_base_resource_ids,
                    stat_server_resource=stat_bs_release_tar_resource_id,
                    meta_server_resource=meta_bs_release_tar_resource_id,
                    prepare_stat_dplan=True,
                    use_requestlog=True,
                    requestlog_resource=request_log_resource_id,
                    cache_daemon_stub_resource=cachedaemon_dump_id,
                    update_parameters_resources=load_shoot_settings_resource_id,
                    stat_shards=[int(shard_num)],
                    meta_mode=meta_mode,
                    specify_cluster_set_config=True,
                    shoot_request_limit=30000,
                    failure_limit=2,
                    subtasks_with_corrections_count=0,
                    subtasks_without_corrections_count=16,
                    stat_store_request_log=True,
                    split_handlers=SPLIT_HANDLERS_BY_META_MODE[meta_mode],
                    __requirements__={'tasks_resource': task.Requirements.tasks_resource},
                ).enqueue().id

    return stat_load_baseline_tasks


@stage(provides='meta_load_baseline_tasks')
def launch_meta_load_baseline(
        task,
        stat_bs_release_tar_resource_id,
        meta_bs_release_tar_resource_id,
        meta_load_request_log_resource_id_map,
        stat_base_tags_map,
        stat_binary_base_resource_id_by_tag,
        meta_base_tags_map,
        meta_binary_base_resource_id_by_tag,
        meta_load_shard_nums,
        meta_load_shard_keys,
        meta_load_shoot_settings_resource_id,
        use_default_debug_cookie=True,
        shoot_request_limit=100000,
        shoot_threads=16,
):
    meta_load_baseline_tasks = {}

    # Check if meta load is enabled inside the function
    # to always fill "meta_load_baseline_tasks" context field.
    # Otherwise "get_spec_data" stage will fail
    # because of absence of "meta_load_baseline_tasks" in context
    if task.Parameters.meta_load:
        for meta_mode, cluster in CLUSTER_BY_META_MODE.items():
            request_log_resource_id = meta_load_request_log_resource_id_map[meta_mode]
            request_log_resource = sdk2.Resource[request_log_resource_id]
            cachedaemon_dump_resource_id = request_log_resource.cachedaemon_dump_res_id

            stat_binary_base_resource_ids = get_base_resource_ids(stat_base_tags_map, stat_binary_base_resource_id_by_tag, meta_load_shard_keys, cluster, server_mode='stat')
            meta_binary_base_resource_ids = get_base_resource_ids(meta_base_tags_map, meta_binary_base_resource_id_by_tag, meta_load_shard_keys, cluster, server_mode='meta')

            meta_load_baseline_tasks[meta_mode] = YabsServerPrepareStatStub(
                task,
                description='prepare stat stub {}, {}'.format(meta_mode, task.Parameters.description),
                tags=task.Parameters.tags,
                use_tmpfs=False,
                ram_space=240,
                stat_server_resource=stat_bs_release_tar_resource_id,
                meta_server_resource=meta_bs_release_tar_resource_id,
                meta_mode=meta_mode,
                specify_cluster_set_config=True,
                stat_shards=map(int, meta_load_shard_nums),
                stat_binary_base_resources=stat_binary_base_resource_ids,
                meta_binary_base_resources=meta_binary_base_resource_ids,
                cache_daemon_stub_resource=cachedaemon_dump_resource_id,
                requestlog_resource=request_log_resource_id,
                update_parameters_resource=meta_load_shoot_settings_resource_id,
                use_default_debug_cookie=use_default_debug_cookie,
                headers_update_dict={},
                shoot_request_limit=shoot_request_limit,
                shoot_threads=shoot_threads,
                __requirements__={'tasks_resource': task.Requirements.tasks_resource},
            ).enqueue().id

    return meta_load_baseline_tasks


@stage(provides='generate_linear_models_service_data_task_id')
def launch_generate_linear_models_service_data(
        task,
        mysql_archive_resource_id, cs_input_spec_resource_id, cs_settings_archive_resource_id, meta_bs_release_yt_resource_id
):
    generate_linear_models_service_data_task = sdk2.Task['YABS_SERVER_GENERATE_LINEAR_MODELS_SERVICE_DATA'](
        task,
        description='Generate data for linear models service, oneshot spec generation',
        input_spec=cs_input_spec_resource_id,
        settings_archive=cs_settings_archive_resource_id,
        mysql_archive_contents=mysql_archive_resource_id,
        bs_release_yt_resource=meta_bs_release_yt_resource_id,
        do_not_restart=True,
        tags=task.Parameters.tags + [task_run_type.SANDBOX_TASK_TAGS['create_oneshot_spec']],
        import_result_ttl=7,
    ).enqueue()
    return generate_linear_models_service_data_task.id


@stage(provides='linear_models_data_resource_id')
def get_linear_models_data(task, generate_linear_models_service_data_task_id):
    check_tasks(task, generate_linear_models_service_data_task_id)

    resource = sdk2.Task[generate_linear_models_service_data_task_id].Parameters.linear_models_service_data_resource
    return resource.id
