import concurrent.futures
import logging
import multiprocessing
import os
import re
from collections import defaultdict, OrderedDict

from sandbox.projects.yabs.qa.module_base import ModuleBase
from sandbox.projects.yabs.qa.sut.bases_provider.adapter_compatible import unpack
from sandbox.projects.yabs.qa.utils.base import BaseIdentityKey

logger = logging.getLogger('BaseStateGenerator')

BASE_NAME_PATTERN = r'\d+\.[0-9a-z_]+\.yabs'


class BaseStateGenerator(ModuleBase):
    def __init__(
        self,
        adapter,
        unpack_workers,
        sync_resource_workers=8,
        core_count=multiprocessing.cpu_count(),
    ):
        ModuleBase.__init__(self, adapter)
        self._unpack_workers = unpack_workers
        self._sync_resource_workers = sync_resource_workers
        self._core_count = core_count

        self._base_state = defaultdict(OrderedDict)

    @property
    def base_state(self):
        return self._base_state

    def get_bases(self, resource_ids):
        sync_resource_futures = []
        sync_resource_executor = concurrent.futures.ThreadPoolExecutor(max_workers=self._sync_resource_workers)
        for resource_id in resource_ids:
            resource_attrs = self.adapter.get_resource_attributes(resource_id)
            base_identity_key = BaseIdentityKey(resource_id=resource_id, use_packed=self.adapter.parameters.use_packed_bases)
            if all([self.base_state.setdefault(resource_attrs['tag'], {}), self.base_state[resource_attrs['tag']].setdefault(base_identity_key, set())]):
                continue

            base_path = self.adapter.get_resource_path(resource_id)
            base_name = '.'.join(os.path.basename(base_path).split('.')[:-1])
            if not bool(re.match(BASE_NAME_PATTERN, base_name)):
                raise Exception('Incorrect base_name - {}, path - {}'.format(base_name, base_path))
            self.base_state[resource_attrs['tag']][base_identity_key].add(os.path.join(os.path.abspath(self.adapter.get_base_dir()), base_name))

            f = sync_resource_executor.submit(self.adapter.sync_resource, resource_id)
            f.tag = resource_attrs['tag']
            f.base_identity_key = base_identity_key
            sync_resource_futures.append(f)

        unpack_base_futures = []
        unpack_base_executor = concurrent.futures.ThreadPoolExecutor(max_workers=self._unpack_workers)
        for f in concurrent.futures.as_completed(sync_resource_futures):
            packed_base_path = f.result()
            unpack_future = unpack_base_executor.submit(
                unpack,
                packed_base_path,
                next(iter(self.base_state[f.tag][f.base_identity_key])),
                self.adapter.get_transport_resource_path(),
                self.adapter.parameters.use_packed_bases,
                core_count=self._core_count,
            )
            unpack_base_futures.append(unpack_future)

        concurrent.futures.wait(unpack_base_futures)
        return self.base_state
