from __future__ import print_function

import shutil
import logging
import tempfile
import time
import os
import errno
import re
import urllib2
import textwrap
import subprocess
import jinja2
from functools import partial

from signal import SIGINT, SIGABRT, SIGKILL, SIGSEGV

from sandbox import sdk2

from sandbox.sandboxsdk.paths import make_folder

from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import network
from sandbox.common.errors import TaskFailure, TaskError

from sandbox.projects.common.search.components import SearchComponent
from sandbox.projects.common import file_utils as fu

from sandbox.projects.yabs.infra.config_template.scatter_source import ENDPOINTS_BY_TAG_SANDBOX, get_scatter_source
from sandbox.projects.yabs.qa.performance.stats.access import IntervalMetrics, LineFormat, dump_ext_metrics
from sandbox.projects.yabs.qa.performance.memory_utils.smaps import read_smaps

from sandbox.projects.yabs.qa.sut.constants import MSAN_FAILURE_RETCODE


logger = logging.getLogger(__name__)

DEMANGLE_CMD = 'c++filt'


def cmd_exists(cmd):
    return any(os.access(os.path.join(path, cmd), os.X_OK) for path in os.environ["PATH"].split(os.pathsep))


def demangle_perf_file(datafile='perf.data'):
    try:
        if not cmd_exists(DEMANGLE_CMD):
            logging.info('Trying to demangle using c++filt')
            demangle_fd, demangle_path = tempfile.mkstemp()
            with open(datafile, 'r') as perf_data_file:
                process.run_process(
                    [DEMANGLE_CMD],
                    wait=True,
                    stdin=perf_data_file,
                    stdout=demangle_fd
                )
            os.close(demangle_fd)
            shutil.copy(demangle_path, datafile)
            os.remove(demangle_path)
        else:
            logging.warning('Oops, unable to find command %s for demangling' % DEMANGLE_CMD)
    except Exception:
        logging.error("Unable to demangle: %s" % datafile, exc_info=True)


class PathMaker(object):
    """PathMaker for YabsServerBase"""

    def __init__(self, server_res_path, work_dir, logs_dir, base_dir=None):
        """
        server_res_path should have the following contents:

        server_res_path/yabs-server
        server_res_path/conf/yabs-*                   # phantom configs
        server_res_path/sandbox-testmode/yabs-*.conf  # testmodes
        server_res_path/llvm-symbolizer
        """

        self.__server_res_path = os.path.abspath(str(server_res_path))
        self.__work_dir = os.path.abspath(str(work_dir))
        self.__base_dir = os.path.abspath(str(base_dir))
        self.__logs_dir = os.path.abspath(str(logs_dir))

        for d in self.__work_dir, self.__base_dir, self.__logs_dir:
            make_folder(d, delete_content=False)

    def server(self, *path_elements):
        return os.path.join(self.__server_res_path, *path_elements)

    @property
    def base_dir(self):
        return self.__base_dir

    def log(self, *path_elements):
        return os.path.join(self.__logs_dir, *path_elements)

    def work(self, *path_elements):
        return os.path.join(self.__work_dir, *path_elements)


class YabsCompatibleBackend(object):
    def get_provided_services(self):
        """Return dict {server:port} which is used to query these services."""
        raise NotImplementedError("Class is abstract")

    def get_provided_ext_tags(self):
        """Return set {tag_1...tag_n}."""
        raise NotImplementedError("Class is abstract")


class YabsServerBase(YabsCompatibleBackend, SearchComponent):
    """
    An interface compatible with DolbilkaExecutor.run_session.

    Basic logic of SearchComponent suits us (but SearchExecutableComponent doesn't)
    """

    name = 'unknown_yabs_server'
    START_WAIT_TIMEOUT = 600
    outputs_to_one_file = False

    LOG_ACCESS = 'access'
    LOG_REQUEST = 'request'
    LOG_EXT = 'ext'
    LOG_PHANTOM = 'phantom'
    IO_BASE = 'io_base'

    LOG_TYPES = [LOG_ACCESS, LOG_REQUEST, LOG_EXT, LOG_PHANTOM, IO_BASE]

    def __init__(
        self,
        path_maker,
        task_instance=None,
        config='yabs-metapartner',
        config_mode='',
        testmode='yabs-meta',
        instance_suffix='',
        args=None,
        store_phantom_logs=(LOG_PHANTOM,),
        custom_env='',
        ping_host='bs.yandex.ru',  # FIXME discuss stupid /ping host limitation
        run_perf=False,
    ):
        """
        :param config: either absolute path or realtive to server_res_path/conf
        :param testmode: either absolute path or realtive to server_res_path/sandbox-testmode
        :param store_phantom_logs: phantom logs to store in logs directory (this is TASK_LOGS inside task)
        :param custom_env: custom env in key1=val1\nk2=val2\n... format, keys and values are stripped of spaces
        """

        task_instance = task_instance or sdk2.Task.current
        super(YabsServerBase, self).__init__(task=task_instance)
        self.process_parameters = []
        self.instance_name = '{}{}'.format(self.name, instance_suffix)
        self.log = logging.getLogger().getChild(self.instance_name)
        self.port = self._get_port()  # Needed for common.dolbilka

        self._ping_host = ping_host

        self.rss_mem_after_start = None
        self.rss_mem_before_stop = None
        self._phantom_log_paths = dict()

        self.__path_maker = path_maker

        self._llvm_symbolizer_path = path_maker.server('llvm-symbolizer')

        # Prepare paths
        self.binary_path = path_maker.server('yabs-server')
        self.work_dir = path_maker.work()

        self.db_list_path = path_maker.log(self.instance_name + '.dblist')

        self.log.debug(
            "Contents of binary bases directory %s:\n%s",
            path_maker.base_dir,
            '\n'.join(os.listdir(path_maker.base_dir))
        )

        for log_dir in ['log', 'log.ft', 'log.rtmr', 'log.ev']:
            make_folder(path_maker.work(log_dir), delete_content=False)
        self.event_dir = self.work_dir

        # Read and patch config
        self.config_path = path_maker.server(config_mode, config)
        logger.debug("Use config {}".format(self.config_path))
        self.config_templated_path = self.config_path + '.temp'

        # Setup env and check config
        self.server_env = self._generate_env(testmode, store_phantom_logs, custom_env)
        self.server_env_str = self._dict_to_str(self.server_env)
        self.args = args or []

        self._render_jinja()
        self.check_result_path = self._check_config()
        self.usage_impacts = []

        # perf profiling
        self.run_perf = run_perf
        self.perf_pid = None

    def _render_jinja(self):
        jinja_env = jinja2.Environment(
            loader=jinja2.FileSystemLoader(os.path.dirname(self.config_path)),
            lstrip_blocks=True,
            trim_blocks=True
        )
        logging.debug("Config path: %s", self.config_path)
        scatter_source = partial(get_scatter_source, tags_dict=ENDPOINTS_BY_TAG_SANDBOX, dc='sandbox')
        tmp = jinja_env.get_template(os.path.basename(self.config_path)).render(self.server_env, get_scatter_source=scatter_source)
        fu.write_file(self.config_templated_path, tmp)

    def _generate_env(self, testmode, store_phantom_logs, custom_env=''):
        """
        Configure yabs-server environment variables
        """
        if testmode:
            if os.path.isabs(testmode):
                testmode_path = testmode
            else:
                testmode_path = self.__path_maker.server('sandbox-testmode', testmode)
            env = _process_testmode(testmode_path, True)
        else:
            env = {}
        env.update(self._get_env_updates())
        env['hostname'] = '"localhost"'  # our echo implementation is lame

        # FIXME BSSERVER-679 put mime_types into BS_RELEASE_TAR and wait until release
        mime_types_path = self.__path_maker.log('mime.types')
        fu.write_file(mime_types_path, _MIME_TYPES)

        env_paths = {
            'base_dir': self.__path_maker.base_dir,
            'event_dir': self.event_dir,
            'ctl_file': self.db_list_path,

            'mime_types_filename': mime_types_path,

            'css_handler_root': self.__path_maker.work(),  # http://host:<monitor3_port>/css
            'run_handler_root': self.__path_maker.log(),       # /run
            'status_handler_root': self.__path_maker.log(),    # /status handler, not interesting
        }

        for log_type in self.LOG_TYPES:
            log_path = (self.__path_maker.log if log_type in store_phantom_logs else self.__path_maker.work)(
                '{}-{}.log'.format(log_type, self.instance_name)
            )
            env_paths[log_type + '_log_file'] = log_path
            self._phantom_log_paths[log_type] = log_path

        env_paths.update(self._get_env_paths())

        for key, val in env_paths.iteritems():
            env[key] = '"{}"'.format(val)

        env.update(self._get_env_for_symbolizer)

        for line in custom_env.splitlines():
            if line.strip():
                key, val = line.split('=', 1)
                env[key.strip()] = val.strip()
        env.update({
            "stat": [],
            "max_stats_in_shard": 0,
            "bigb_clusters": ["sandbox"],
            "rsya_hit_models_light_services": ["yp_prod"],
            "rsya_hit_models_heavy_services": ["yp_prod"],
        })
        return env

    def _dict_to_str(self, env):
        str_env = {str(key): str(val) for key, val in env.iteritems()}

        with open(self.__path_maker.log(self.instance_name + '.env'), 'w') as envdump:
            envdump.write('\n'.join('='.join(kv) for kv in str_env.iteritems()))
        return str_env

    def _get_port(self):
        """
        Returns the very port on which the component receives data.
        """
        raise NotImplementedError("Class is abstract")

    def _get_monitor2_port(self):
        """
        Return monitor2 port (or None if no monitor2 exists)
        """
        raise NotImplementedError("Class is abstract")

    def _get_listen_ports(self):
        """
        Return list of all ports on which this component listens
        """
        raise NotImplementedError("Class is abstract")

    def _get_env_updates(self):
        """
        Return updates to server environment variables
        """
        raise NotImplementedError("Class is abstract")

    def _get_env_paths(self):
        """
        Return paths to insert into to server environment variables
        """
        raise NotImplementedError("Class is abstract")

    def get_bin_db_list(self, timeout=3.9):
        """
        Obtains list of binary base filenames from ctl_file.
        """
        if not self.is_running():
            self._raise_error("cannot get binary base list: process is not running")

        deadline = time.time() + timeout
        pause = 0.25
        while True:
            try:
                db_list_file = open(self.db_list_path)
            except IOError as err:
                if err.errno != errno.ENOENT:
                    raise
                now = time.time()
                if now > deadline:
                    state_str = 'is STILL RUNNING' if self.is_running() else 'HAS DIED'
                    self._raise_error(
                        "file {} is still missing after {.1f} seconds and process {}".format(self.db_list_path, now - deadline, state_str)
                    )
                time.sleep(pause)
                pause *= 1.5
            else:
                return db_list_file.read().splitlines()

    def wait_ping(self, timeout=840):
        """
        Wait until /ping stops responding with codes 503/523
        """
        if not self.is_running():
            self._raise_error("Will not wait for ping: process is not running")

        self._wait_for_connect(self.START_WAIT_TIMEOUT)

        deadline = time.time() + timeout
        pause = 0.25
        request = urllib2.Request("http://localhost:{}/ping".format(self.port), headers={"Host": self._ping_host})

        expected_error_codes = [
            503,
            523,  # Current Base Unknown
        ]

        while True:
            error_msg = ''
            try:
                urllib2.urlopen(request, timeout=10)
            except urllib2.HTTPError as e:
                if e.code not in expected_error_codes:
                    self._raise_error("Ping failed with " + str(e))
                error_msg = "/ping returned %d" % e.code
            except Exception as e:
                error_msg = "/ping exception: %s" % (str(e))
            else:
                break

            now = time.time()
            if now > deadline:
                self._raise_error(
                    "No 200 response after %s seconds (port %s): %s" % (timeout, self.port, error_msg)
                )
            logger.debug(error_msg)
            time.sleep(pause)
            pause = 2.0
            logger.debug("Retrying /ping ...")

        return

    def get_phantom_log_path(self, log_type):
        return self._phantom_log_paths[log_type]

    def get_bin_db_dir(self):
        return self.__path_maker.base_dir

    def check_access_log(
        self,
        ext_sharded=None,
        quantiles=(90, 99),
        ignore_handlers=None,
        ignore_ext=None,
        error_rate_thr=0.005,
        request_error_rate_thr=0.005,
        ext_error_rate_thr=0.005,
        ext_service_thresholds=None
    ):
        """
        Calculate access log metrics, dump aggregates into log file and check error rates.

        :param ext_sharded: list of external requests types for which per-shard metrics should be calculated
        :param quantiles: response tiem percentiles to calculate
        :param ignore_handlers: ignore errors on these handlers
        :param ignore_ext: ignore errors on these external request types
        :param error_rate_thr: threshold 5xx rate (per-handler), raise exception if exceeded
        :param request_error_rate_thr: threshold 4xx rate
        :param ext_error_rate_thr: threshold external requests rate (per type of external request)
        :param ext_service_thresholds: thresholds for particular external services

        :return: total request count in access log
        """
        ignore_handlers = ignore_handlers or {}
        ignore_ext = ignore_ext or {}
        ext_service_thresholds = ext_service_thresholds or {}
        aggr = self.aggregate_access_log(ext_sharded)

        def check(errors_dict, times_dict, ignore, rate_thr, prefix, service_thresholds=None):
            service_thresholds = service_thresholds or {}
            bad_keys = []
            for key, err_count in errors_dict.iteritems():
                if key not in ignore:
                    discounted_err_count = max(0, err_count - 3 * err_count**0.5)
                    req_count = len(times_dict.get(key, []))
                    if discounted_err_count > req_count * service_thresholds.get(key, rate_thr):
                        bad_keys.append('.'.join((prefix, key)))
            return bad_keys

        for name, errors in aggr.ext_errors.iteritems():
            requests = len(aggr.ext_times.get(name, []))
            logging.debug('External service %s had %s rate of errors (%s out of %s failed)', name, str(errors * 1. / requests), str(errors), str(requests))

        ext_error_keys = check(aggr.ext_errors, aggr.ext_times, ignore_ext, ext_error_rate_thr, 'ext_errors', ext_service_thresholds)
        error_keys = check(aggr.errors, aggr.tot_times, ignore_handlers, error_rate_thr, 'errors')
        request_error_keys = check(aggr.request_errors, aggr.tot_times, ignore_handlers, request_error_rate_thr, 'request_errors')

        metrics = aggr.prepare_metrics(quantiles)
        ext_metrics = aggr.prepare_ext_statistics(quantiles)
        self._dump_access_log_metrics(metrics, ext_error_keys + error_keys + request_error_keys)
        self._dump_access_log_ext_stats(ext_metrics)

        bad_keys = []

        if request_error_keys:
            errors_category = 'request (4xx) errors'
            bad_keys = request_error_keys
        elif error_keys or ext_error_keys:
            errors_category = '5xx/ext errors'
            bad_keys = error_keys + ext_error_keys
        if bad_keys:
            self._raise_error(
                "Too many {} errors in our access log:\n{}".format(
                    errors_category,
                    '\n'.join('{}: {}'.format(k, metrics[k]) for k in bad_keys),
                )
            )

        return aggr.n_requests

    def aggregate_access_log(self, ext_sharded):
        """
        Returns instance of IntervalMetrics
        """
        aggr = IntervalMetrics(only_ext_sharded=ext_sharded)
        access_log_path = self.get_phantom_log_path(self.LOG_ACCESS)
        with open(access_log_path, 'r') as log:
            for line in log:
                if line.startswith('#'):
                    fmt = LineFormat(line)
                else:
                    try:
                        aggr.add_line(line.split('\t'), fmt)  # FIXME We'll get NameError in case of bad first line
                    except ValueError:  # FIXME this is a hole
                        self.log.warning("Bad line in access log:\n%s", line)
        return aggr

    def _dump_access_log_metrics(self, metrics, bad_names):
        metrics_file_name = self.__path_maker.log(self.name + '-accesslog-metrics.txt')
        bad_list = sorted(list(bad_names))
        good_list = sorted(list(set(metrics) - set(bad_names)))
        with open(metrics_file_name, 'w') as metrics_file:
            for key in bad_list:
                print('{}: {} !!BROKEN!!'.format(key, metrics.get(key, 'INTERNAL PARSING ERROR')), file=metrics_file)
            for key in good_list:
                print('{}: {}'.format(key, metrics[key]), file=metrics_file)

    def _dump_access_log_ext_stats(self, ext_metrics):
        if not ext_metrics:
            return

        metrics_file_name = self.__path_maker.log(self.name + '-accesslog-ext-metrics.txt')
        with open(metrics_file_name, 'w') as metrics_file:
            metrics_file.write(dump_ext_metrics(ext_metrics))

    def _get_process_and_command_line(self, command_option, stdout, stderr):
        command_line = [self.binary_path, command_option, self.config_templated_path] + self.args
        return subprocess.Popen(command_line, env=self.server_env_str, stdout=stdout, stderr=stderr), command_line

    def _check_config(self):
        with sdk2.helpers.ProcessLog(self.task, logger=self.instance_name + '.config_check') as process_log:
            process_log.logger.propagate = True
            with open(self.__path_maker.log(self.instance_name + '.config_check'), 'w') as f_out:
                check_process, command_line = self._get_process_and_command_line(
                    'check',
                    stdout=f_out,
                    stderr=process_log.stderr
                )

                self._try_register_process(check_process, command_line)
                check_process.communicate()
                if check_process.returncode:
                    process.throw_subprocess_error(check_process)

    def _try_register_process(self, process, command_line):
        process_pid = process.pid
        try:
            sdk2.helpers.ProcessRegistry.register(process_pid, command_line)
        except AttributeError:
            self.log.warning('Could not register process for coredump collection, we must be running in standalone environment')

    def get_manual_cmdline(self):
        return ' '.join([self.binary_path, 'run', self.check_result_path] + self.args)

    def start(self):
        """
        Start process and exit
        """
        self._process_log_context = sdk2.helpers.ProcessLog(self.task, self.instance_name + '.run')
        self._process_log_context.__enter__()
        self._wait_free_listen_ports()
        self.process, command_line = self._get_process_and_command_line(
            'run',
            stdout=self._process_log_context.stdout,
            stderr=self._process_log_context.stderr
        )
        self._try_register_process(self.process, command_line)
        self.log.info('started process with pid: %s', self.process.pid)

    def wait(self):
        """
        Wait and do pre-start checks
        """
        self.log.info(
            "Wait for %s seconds until process %s starts to listen",
            self.START_WAIT_TIMEOUT, self.process.pid
        )
        self._wait_for_connect(self.START_WAIT_TIMEOUT)
        self.rss_mem_after_start = self._get_rss_memory_noexcept()

    def is_running(self):
        """
            Check if the process is running.
            Fail the task if it was killed by something bad (SEGV/KILL/ABORT)
        """
        if not self.process:
            return False

        retcode = self.process.poll()
        if retcode is None:
            return True

        self.log.info("process exited with code %s", retcode)
        if retcode in (-SIGABRT, -SIGKILL, -SIGSEGV, MSAN_FAILURE_RETCODE):
            self._process_post_mortem()
            raise Exception("We should never get here")
        return False

    def _wait_for_connect(self, timeout):
        """
        Wait until we are able to connect to all ports on which process listens.
        """

        finish = time.time() + timeout
        while time.time() < finish:
            if not self.is_running():
                self._process_post_mortem()
                raise Exception("We should never get here.")
            if all([not network.is_port_free(port) for port in self._get_listen_ports()]):
                self.log.info("something is listening on all ports, we are ready")
                return True
            time.sleep(1)
        self.log.error("Waiting for start of process %s timed out", self.process.pid)
        if self.is_running():
            self.log.info("Kill process %s", self.process.pid)
            os.kill(self.process.pid, 6)  # generate coredump
        self._raise_error("connecting to some of the ports timed out", exc=TaskFailure)

    def _process_post_mortem(self):
        process.throw_subprocess_error(self.process)

    def use_component(self, work_func):
        usage_impact = {'start': self._get_usage_data_noexcept()}
        self.log.info("start use_component")
        try:
            self.wait_ping()
            if self.run_perf:
                self.log.info("before perf_record_start")
                self.perf_record_start()
            return work_func()
        finally:
            usage_impact['stop'] = self._get_usage_data_noexcept()
            self.usage_impacts.append(usage_impact)

            if self.is_running():
                self.log.info("still running after use")
            else:
                self.log.info("died during use")
                self._process_post_mortem()
            if self.run_perf:
                self.perf_record_stop()

    def stop(self):
        """
        Stop process and execute checks
        """
        self.log.info('Stopping process')

        if self.process is None:
            self._raise_error("process was not started (bug in the task??)")
        if not self.is_running():
            self.log.warning("process died")
            self._process_post_mortem()
        else:
            self.dump_monitor2_data()

        component_stderr = str(self._process_log_context.stderr.path)

        self.rss_mem_before_stop = self._get_rss_memory_noexcept()

        self.process.kill()
        self.process.wait()
        time.sleep(1)

        self._wait_free_listen_ports()
        self.process = None
        logging.info('Stopped %s', self.name)

        if component_stderr:
            logging.info('Verifying stderr for "%s" at "%s"', self.name, component_stderr)
            self.verify_stderr(component_stderr)
        else:
            logging.error('Empty component output filename, stderr verification disabled for "%s"', self.name)
        self._process_log_context.__exit__(None, None, None)
        self._process_log_context = None

    def __enter__(self):
        return super(YabsServerBase, self).__enter__()

    def __exit__(self, exc_type, exc_value, traceback):
        try:
            if exc_type is None:
                self.stop()
            else:
                # We do not want to override exceptions from "with" block!!
                try:
                    self.stop()
                except BaseException:
                    self.log.exception("Failed to stop yabs-server:")
        finally:
            if self._process_log_context:
                self._process_log_context.__exit__(exc_type, exc_value, traceback)
                self._process_log_context = None

    def dump_monitor2_data(self):
        monitor2_port = self._get_monitor2_port()
        if monitor2_port is None:
            # FIXME: use less mysterious (but PRECISE) word instead of "monitor2"
            logging.info("Not dumping monitor2 data: there is no monitor2 port")
            return
        monitor2_url = "http://localhost:{}".format(monitor2_port)
        try:
            monitor2_data = urllib2.urlopen(monitor2_url, timeout=10).read()
        except Exception:  # pylint: disable=W0703
            self.log.warning("Failed to obtain monitoring data from monitor2 port: %s", monitor2_port, exc_info=True)
        else:
            # FIXME: fix overwrites of these files BSSERVER-6406
            fu.write_file(
                self.__path_maker.log('{}.monitor2.json'.format(self.instance_name)),
                monitor2_data
            )

    def verify_stderr(self, file_name):
        """
            Check stderr output stored in file for errors/warnings existance.

            To be redefined in child classes.
            Also you may want separate stderr/stdout streams to different files using
            self.outputs_to_one_file = False
        """
        pass

    def get_pid(self):
        if not self.is_running():
            self._raise_error("process is not running")
        return self.process.pid

    def get_rss_memory(self):
        """
            Get resident memory size (in Kbytes)
        """
        with open("/proc/{}/status".format(self.get_pid())) as f:
            for line in f:
                if line.startswith('VmRSS:'):
                    size_kb = line[6:].strip('\n').strip(' ')
                    if size_kb.endswith(' kB'):
                        return int(size_kb[:-3])

    def _get_rss_memory_noexcept(self):
        try:
            return self.get_rss_memory()
        except Exception:  # pylint: disable = W0703
            self.log.exception("Failed to get RSS (resident set size)")
        return None

    def get_usage_data(self):
        """
            Return dict with various resource usage data at current moment.
            Method is usually used in use_compnent before use and after use.
        """
        smaps = read_smaps(self.get_pid())
        anon_total = 0
        ahp_total = 0

        non_hp_maps_anon_total = 0

        for s in smaps:
            flags = s['VmFlags'].split()
            if 'hg' in flags:
                anon_total += int(s['Anonymous'])
                ahp_total += int(s['AnonHugePages'])
            else:
                non_hp_maps_anon_total += int(s['Anonymous'])
        return {
            "AHP_fraction": 1.0 * ahp_total / anon_total if ahp_total else 0,
            "non_HP_maps_anon_total": non_hp_maps_anon_total,
        }

    def _get_usage_data_noexcept(self):
        try:
            return self.get_usage_data()
        except Exception:  # pylint: disable = W0703
            self.log.exception("Failed to get resource usage data")
        return None

    def _wait_free_listen_ports(self, timeout=60):
        """
        Wait until all listen ports are free
        """
        for port in self._get_listen_ports():
            if not network.wait_port_is_free(port, timeout=timeout):
                self._raise_error('Port {} is not free.'.format(port))

    def _raise_error(self, msg, exc=TaskError):
        raise exc("{}: {}".format(self.instance_name, msg))

    def perf_record_start(self, datafile='perf.data'):
        cmdline = ['perf', 'record', '-p', str(self.process.pid), '-g', '-F', '99', '-o', datafile]
        self.perf_pid = process.run_process(
            cmdline,
            wait=False,
        ).pid
        self.log.info("end perf_record_start")

    def perf_record_stop(self, datafile='perf.data'):
        if self.perf_pid:
            logging.info('Send SIGINT to perf')
            os.kill(self.perf_pid, SIGINT)
            os.waitpid(self.perf_pid, 0)
            demangle_perf_file(datafile)
            self.perf_pid = None

    @property
    def _get_env_for_symbolizer(self):
        return {
            'ASAN_SYMBOLIZER_PATH': self._llvm_symbolizer_path,
            'MSAN_SYMBOLIZER_PATH': self._llvm_symbolizer_path,
        }


def _process_testmode(testmode_tail_path, convert_hosts_to_localhost=False):
    env = {}
    with open(testmode_tail_path) as testmode_tail:
        for line_no, line in enumerate(testmode_tail):
            if not line.strip():
                continue
            m = re.match(r'^(\S+)\s+(\S+)\s+(.*)$', line)
            if m is None:
                raise TaskFailure(
                    "Bad line {} in testmode file {}:\n{}".format(
                        line_no, testmode_tail_path, line
                    ))
            var, cmd, arg = m.groups()
            if cmd == 'echo':
                env[var] = arg
            elif cmd in ['addrs', 'addrs6']:
                hosts = arg.split()
                if not convert_hosts_to_localhost:
                    if any(host != 'localhost' for host in hosts):
                        raise TaskFailure(
                            "Remote hosts on line {} of testmode file {}:\n{}".format(
                                line_no, testmode_tail_path, line
                            ))

                addr = ('127.0.0.1' if cmd == 'addrs' else '::1')
                env[var] = "{{ {} }}".format(' '.join([addr] * len(hosts)))
            else:
                raise TaskFailure(
                    "Bad command {} at line {} of testmode file {}".format(
                        cmd, line_no, testmode_tail_path
                    ))
    return env


_MIME_TYPES = textwrap.dedent("""
    image/gif	gif
    application/x-shockwave-flash	swf
    image/jpeg	jpg jpeg
    image/png	png
    image/bmp	bmp
    audio/mpeg	mp3
    application/x-javascript	js
    text/html	html htm
    text/css	css
    application/xml	xml
    application/x-flash-video	flv
    text/plain	txt
    application/octet-stream	swc
    image/svg+xml	svg
""")
