import os
import logging
import shutil
import tarfile
from sandbox import sdk2
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common.search.components import component as components_common
from sandbox.projects.yabs.qa.sut.utils import reserve_port
from sandbox.projects.yabs.qa.sut.components.base import YabsCompatibleBackend


LOGGER = logging.getLogger(__name__)


WAIT_TIMEOUT_DEFAULT = 600


class YabsLinearModels(
        components_common.ProcessComponentMixinWithShutdownSDK2,
        YabsCompatibleBackend,
        components_common.WaitPortComponentMixin,
        components_common.Component,
):
    """
    Interface for yabs linear model service.

    https://a.yandex-team.ru/arc/trunk/arcadia/search/daemons/begemot/yabs_linear_models
    """

    name = "yabs-linear-models"
    service_tags = {"linear_models_service", "linear_models_service_wide"}

    def __init__(self, task, binary_path, data_path, wait_timeout=WAIT_TIMEOUT_DEFAULT, work_dir=os.curdir):
        self.task = task
        self._port, self._socket = reserve_port()
        self._grpc_port, self._grpc_socket = reserve_port()
        self._binary_path = binary_path
        self._data_path = os.path.abspath(data_path)
        self._unpacked_data_path = os.path.abspath(os.path.join(work_dir, "ylm_data_folder"))

        if os.path.isdir(self._unpacked_data_path):
            logging.debug("Reuse existing yabs_linear_models data %s", self._unpacked_data_path)
        else:
            logging.debug("Extract %s to %s", self._data_path, self._unpacked_data_path)
            with tarfile.open(self._data_path) as tar:
                tar.extractall(path=self._unpacked_data_path)

            # Tell begemot daemon to init YabsLinearModelsMerger rule
            try:
                os.makedirs(os.path.join(self._unpacked_data_path, "YabsLinearModelsMerger"))
            except OSError as e:
                # File exists
                if e.errno == 17:
                    pass

        components_common.WaitPortComponentMixin.__init__(self, endpoints=[("localhost", self._port)], wait_timeout=wait_timeout)
        components_common.ProcessComponentMixinWithShutdownSDK2.__init__(
            self,
            args=map(str, (self._binary_path, "-d", self._unpacked_data_path, "-p", self._port, "-g", self._grpc_port)),
            shutdown_url="http://localhost:{}/admin?action=shutdown".format(self._port),
            log_prefix=self.name,
        )
        LOGGER.info("Component '%s' initialized successfully with port '%s'", self.name, self._port)

    def __enter__(self):
        self.process_log = sdk2.helpers.ProcessLog(self.task, logger=self._log_prefix)
        super(YabsLinearModels, self).__enter__()

    def flush_state(self):
        shutil.rmtree(self._unpacked_data_path)

    def get_port(self):
        return self._port

    def get_provided_services(self):
        res = {}
        for k in self.service_tags:
            res[k] = self._port
        return res

    def get_provided_ext_tags(self):
        return self.service_tags

    def verify_stderr(self, custom_stderr_path=None):
        """Check stderr output stored in file for errors/warnings existence."""
        stderr_path = custom_stderr_path or self.process.stderr_path
        if not os.path.exists(stderr_path):
            return

        LOGGER.info("Verifying stderr path %s ...", stderr_path)
        with open(stderr_path) as stderr:
            for line in stderr:
                if "[ERROR]" in line:
                    eh.check_failed("Unexpected stderr log line: '{}'".format(line))
