import logging
import os

from sandbox.projects.yabs.qa.sut.components.base import YabsServerBase
from sandbox.projects.yabs.qa.sut.constants import DEFAULT_STAT_SHARD
from sandbox.projects.yabs.qa.sut.utils import reserve_port
from sandbox.projects.yabs.qa.utils.general import makedirs_except

LAST_STAT_SHARD = 18  # FIXME No longer nneded with stat_enable?


class YabsMeta(YabsServerBase):

    name = 'yabs-meta-unknown'

    def __init__(
        self,
        path_maker,
        surf_data_path,
        task_instance=None,
        static_data_path=None,
        config='yabs-metapartner',
        config_mode='',
        testmode='yabs-meta',
        listen_port=None,
        monitor2_port=None,
        instance_suffix='',
        backends=None,
        custom_env='',
        ping_host='yabs.yandex.ru',
        store_phantom_logs=(YabsServerBase.LOG_PHANTOM,),
    ):
        self.name, _ = os.path.splitext(config)

        self.static_data_path = path_maker.server()  # And hope nobody asks /resource
        if static_data_path:
            self.static_data_path = os.path.abspath(static_data_path)

        self.surf_data_path = os.path.abspath(surf_data_path)
        self.service_discovery_cache_path = os.path.join(path_maker.work("service_discovery_cache"), 'meta')
        makedirs_except(self.service_discovery_cache_path)

        self.backends = backends or []

        self.sockets = []
        self.misc_ports = {}

        self.listen_port, socket = listen_port if listen_port is not None else reserve_port()
        self.sockets.append(socket)
        self.dsp_port = self.lock_free_port()
        misc_ports_list = [
            'metarank_proxy_backend',
            'metapartner_proxy_backend',
            'metasearch_proxy_backend',
        ]
        for port_name in misc_ports_list:
            self.add_misc_port(port_name)
        self.monitor_ports = []
        self.monitor_sockets = []
        for port, socket in [reserve_port() for i in range(4)]:
            self.monitor_ports.append(port)
            self.monitor_sockets.append(socket)
        if monitor2_port is not None:
            self.monitor_ports[1] = monitor2_port

        self.solomon_stats_meta_port, self.solomon_stats_meta_socket = reserve_port()
        self.solomon_stats_port = self.solomon_stats_meta_port

        super(YabsMeta, self).__init__(
            path_maker,
            task_instance=task_instance,
            config=config,
            config_mode=config_mode,
            testmode=testmode,
            instance_suffix=instance_suffix,
            custom_env=custom_env,
            ping_host=ping_host,
            store_phantom_logs=store_phantom_logs,
        )

    def lock_free_port(self):
        port, socket = reserve_port()
        self.sockets.append(socket)
        return port

    def add_misc_port(self, port_name):
        self.misc_ports[port_name + '_port'] = self.lock_free_port()

    def _get_port(self):
        return self.listen_port

    def _get_monitor2_port(self):
        return self.monitor_ports[1]

    def _get_listen_ports(self):
        return [self.listen_port] + self.monitor_ports[:3]  # FIXME-BSSERVER-13677: add scheduler_monitor_port to listen ports when review 1222227 is in oneshot spec

    def _get_env_updates(self):
        updates = {
            'port': self.listen_port,
            'yabs_dsp_port': self.dsp_port,
            'yabs_dsp_metasearch_port': self.dsp_port,
            'monitor1_port': self.monitor_ports[0],
            'monitor2_port': self.monitor_ports[1],
            'monitor3_port': self.monitor_ports[2],
            'scheduler_monitor_port': self.monitor_ports[3],
            'yabs_to_an_redirect_port': self.listen_port,
            'an_to_yabs_redirect_port': self.listen_port,
            # FIXME thread counts are arbitrary
            'thrs_main': 6,
            'thrs_client': 6,
            'thrs_rank': 15,
            'thrs_code': 15,
            'thrs_async_scatter': 4,
            'thrs_rtb': 6,
            'solomon_stats_meta_port': self.solomon_stats_meta_port,
            'thrs_saas': 1,
        }
        logging.debug('Meta env updates before: %s', updates)
        updates.update(self.misc_ports)
        yabstat_shards = set()

        for backend in self.backends:
            for service, port in backend.get_provided_services().iteritems():
                updates['{}_port'.format(service)] = port
                if service.startswith('yabstat'):
                    shard = service.split('yabstat')[1]
                    yabstat_shards.add(shard)
                    updates[service] = '{ %s }' % ' '.join(['127.0.0.1'] * 32)

        yabstat_shards = yabstat_shards or {DEFAULT_STAT_SHARD}  # Meta doesn't start with all stats disabled. FIXME remove this workaround.
        updates['stat_enable'] = '{ %s }' % ' '.join(sorted(yabstat_shards))

        # FIXME stat_disable is needed for obsolete yabs-server only
        disabled_shards = set('%02d' % s for s in range(1, LAST_STAT_SHARD + 1)) - yabstat_shards
        updates['stat_disable'] = '{ %s }' % ' '.join(sorted(disabled_shards))

        logging.debug('Meta env updates after: %s', updates)
        return updates

    def get_provided_services(self):
        """Meta provides no services known to yabs-server"""
        return {}

    def get_provided_ext_tags(self):
        return {"yabs", "yabs_debug", "get_tsar", "get_saas_data"}

    def _get_env_paths(self):
        return {
            'root_handler_root': self.static_data_path,                                # /
            'media_handler_root': os.path.join(self.static_data_path, 'media'),        # /media
            'resource_handler_root': os.path.join(self.static_data_path, 'resource'),  # /resource
            'yandex_surf_data_file': os.path.abspath(self.surf_data_path),
            'service_discovery_cache': self.service_discovery_cache_path,
        }
