import logging
import os

from base import YabsServerBase
from sandbox.projects.yabs.qa.sut.constants import DEFAULT_STAT_SHARD
from sandbox.projects.yabs.qa.sut.utils import reserve_port
from sandbox.projects.yabs.qa.utils.general import makedirs_except


class YabsStat(YabsServerBase):

    def __init__(
        self,
        path_maker,
        task_instance=None,
        shard=DEFAULT_STAT_SHARD,
        monitor_ports=None,
        instance_suffix='',
        log_requests=False,
        custom_env='',
        store_phantom_logs=(YabsServerBase.LOG_PHANTOM,),
        run_perf=False,
        config='yabs-stat',
        config_mode='',
        backends=None,
        stat_stub_port=None,
    ):
        self.name = 'yabs-stat' + shard
        self.provided_service_name = 'yabstat' + shard

        if stat_stub_port is None:
            self.listen_port, self.socket = reserve_port()
        else:
            self.listen_port, self.socket = stat_stub_port, None
            logging.info('Stat stub will be used for %s, port: %s', self.provided_service_name, stat_stub_port)
            return

        self.monitor_ports = []
        self.monitor_sockets = []
        for port, socket in [reserve_port() for _ in range(4)]:
            self.monitor_ports.append(port)
            self.monitor_sockets.append(socket)

        shard_no = int(shard)

        self.log_requests = log_requests

        self.stat_mgroup = shard_no - 1

        self.backends = backends or []

        self.solomon_stats_stat_port, self.solomon_stats_stat_socket = reserve_port()
        self.solomon_stats_port = self.solomon_stats_stat_port

        self.service_discovery_cache_path = os.path.join(path_maker.work("service_discovery_cache"), 'stat')
        makedirs_except(self.service_discovery_cache_path)

        super(YabsStat, self).__init__(
            path_maker=path_maker,
            task_instance=task_instance,
            config=config,
            config_mode=config_mode,
            testmode='yabs-stat',
            instance_suffix=instance_suffix,
            args=['stat' + shard],
            store_phantom_logs=store_phantom_logs,
            custom_env=custom_env,
            ping_host='yabstat.yandex.ru',
            run_perf=run_perf,
        )

    def _get_port(self):
        return self.listen_port

    def _get_monitor2_port(self):
        return self.monitor_ports[1]

    def _get_listen_ports(self):
        return [self.listen_port] + self.monitor_ports[:3]  # FIXME-BSSERVER-13677: add scheduler_monitor_port to listen ports when review 1222227 is in oneshot spec

    def _get_env_updates(self):
        updates = {
            'stat_mgroup': self.stat_mgroup,
            'port': self.listen_port,
            'monitor1_port': self.monitor_ports[0],
            'monitor2_port': self.monitor_ports[1],
            'monitor3_port': self.monitor_ports[2],
            'scheduler_monitor_port': self.monitor_ports[3],
            'yabstat_loggers': "{ access_logger default_logger }" if self.log_requests else "{ access_logger }",
            'thrs_work': 19,
            'thrs_main': 2,
            'thrs_client': 1,
            'thrs_async_scatter': 1,
            'solomon_stats_stat_port': self.solomon_stats_stat_port,
            'work_limit': 500,
            'thrs_saas': 1,
        }
        for backend in self.backends:
            for service, port in backend.get_provided_services().iteritems():
                updates['{}_port'.format(service)] = port
        return updates

    def _get_env_paths(self):
        return {
            'service_discovery_cache': self.service_discovery_cache_path,
        }

    def get_provided_services(self):
        return {self.provided_service_name: self.listen_port}

    def get_provided_ext_tags(self):
        return {
            "pmatch",
            "content",
            "count",
            "stat_dict",
            "bansearch",
            "audit",
            "ordstat",
            "domcost",
            "rank",
            "write_wide",
            "get_data",
            "get_tsar",
            "get_saas_data"
        }
