import json
import os
import logging

from collections import defaultdict

from sandbox import sdk2
from sandbox.common.fs import get_unique_file_name
from sandbox.projects.yabs.qa.sut.constants import DEFAULT_STAT_SHARD
from sandbox.projects.yabs.qa.sut.components import (
    meta,
    stat,
    turl,
    null_stub,
    base,
    linear_models,
)


logger = logging.getLogger(__name__)


META_MODES = {
    'bs': 'yabs-metapartner',
    'yabs': 'yabs-metasearch',
    'bsrank': 'yabs-metarank',
    'mc': 'yabs-metrika',
}


def get_yabs_server_base_revision(server_path):
    release_info_path = os.path.join(server_path, "release_info")
    logging.debug("Get yabs-server base revision from %s", release_info_path)
    try:
        with open(release_info_path) as f:
            release_info = json.load(f)
    except Exception as e:
        logging.error("Failed to read %s: %s", release_info_path, e)
        return None

    return int(release_info["revision"])


class YabsServerFactoryStandalone(object):
    """NOT THREAD-SAFE AT ALL!"""

    def __init__(
        self,
        server_path,
        surf_data_path,
        task_instance=None,
        work_dir=os.curdir,
        logs_dir=None,
        base_dir=None,
        server_base_revision=None,
        prefix='yabs_server_factory'
    ):
        self._task_instance = task_instance or sdk2.Task.current
        if not logs_dir:
            logs_dir = os.path.join(os.getcwd(), 'logs')
            if not os.path.exists(logs_dir):
                os.makedirs(logs_dir)
        self.__logs_dir = get_unique_file_name(logs_dir, prefix)
        self.__base_dir = base_dir or os.path.abspath('yabs_server_bases')
        self.__work_dir = os.path.abspath(os.path.join(work_dir, prefix))

        self.__server_path = server_path
        logger.debug("Use yabs-server from \"%s\"", self.__server_path)

        self._surf_data_path = surf_data_path

        if not server_base_revision:
            server_base_revision = get_yabs_server_base_revision(self.__server_path)
        self._server_base_revision = server_base_revision
        logger.debug("yabs-server base revision: %s", self._server_base_revision)

        self.__path_maker_counts = defaultdict(int)

    @property
    def server_path(self):
        return self.__server_path

    def create_meta(
        self,
        meta_mode,
        backends=None,
        custom_env='',
        store_access_log=False,
        store_request_log=False,
        config_mode='',
    ):
        store_phantom_logs = self._make_log_list(store_access_log, store_request_log)

        if store_request_log:
            custom_env = 'loggers={access_logger default_logger}\n' + custom_env

        config_name = META_MODES[meta_mode]
        return meta.YabsMeta(
            self._get_path_maker(config_name),
            task_instance=self._task_instance,
            surf_data_path=self._surf_data_path,
            static_data_path=None,
            config=config_name,
            config_mode=config_mode,
            backends=backends,
            custom_env=custom_env,
            ping_host=(meta_mode + '.yandex.ru'),
            store_phantom_logs=store_phantom_logs,
        )

    def create_turl(
        self,
        shard,
        backends=None,
        custom_env='',
        store_access_log=False,
        config_mode='',
    ):
        return turl.YabsTurl(
            self._get_path_maker('turl{}'.format(shard)),
            task_instance=self._task_instance,
            shard=shard,
            backends=backends,
            config_mode=config_mode,
            custom_env=custom_env,
            store_phantom_logs=self._make_log_list(store_access_log),
        )

    def create_stat(
        self,
        shard=DEFAULT_STAT_SHARD,
        log_requests=False,
        custom_env='',
        store_access_log=False,
        run_perf=False,
        config_mode='',
        backends=None,
        stat_stub_port=None,
    ):
        return stat.YabsStat(
            self._get_path_maker('stat{}'.format(shard)),
            task_instance=self._task_instance,
            shard=shard,
            log_requests=log_requests,
            custom_env=custom_env,
            store_phantom_logs=self._make_log_list(store_access_log),
            run_perf=run_perf,
            config_mode=config_mode,
            backends=backends,
            stat_stub_port=stat_stub_port,
        )

    def create_null_stub(self, provided_services):
        return null_stub.YabsNull(
            self._get_path_maker('null_stub'),
            task_instance=self._task_instance,
            provided_services=provided_services,
        )

    LINEAR_MODELS_WAIT_TIMEOUT_DEFAULT = linear_models.WAIT_TIMEOUT_DEFAULT

    def create_linear_models(self, binary_path, data_path, wait_timeout=LINEAR_MODELS_WAIT_TIMEOUT_DEFAULT):
        logger.info('Setup yabs_linear_models service')
        return linear_models.YabsLinearModels(
            self._task_instance,
            binary_path,
            data_path,
            wait_timeout=wait_timeout,
            work_dir=self.__work_dir
        )

    @property
    def server_base_revision(self):
        return self._server_base_revision

    def _get_path_maker(self, prefix=''):
        count = self.__path_maker_counts[prefix]
        self.__path_maker_counts[prefix] += 1

        parts = (prefix,) if not count else (prefix, str(count))

        def _work_subdir(root):
            return os.path.join(root, '_'.join(parts))

        return base.PathMaker(
            server_res_path=self.server_path,
            base_dir=self.__base_dir,
            work_dir=_work_subdir(self.__work_dir),
            logs_dir=_work_subdir(self.__logs_dir),
        )

    @staticmethod
    def _make_log_list(store_access_log=False, store_request_log=False):
        store_phantom_logs = [base.YabsServerBase.LOG_PHANTOM, base.YabsServerBase.IO_BASE]
        if store_access_log:
            store_phantom_logs.append(base.YabsServerBase.LOG_ACCESS)
        if store_request_log:
            store_phantom_logs.append(base.YabsServerBase.LOG_REQUEST)

        return store_phantom_logs
