import logging
import os
import tarfile
from abc import ABCMeta, abstractmethod

from sandbox.common.fs import get_unique_file_name
from sandbox.projects.common.yabs.cachedaemon import CacheDaemonStubSandboxNonIntegrated, SERVICES_BY_TYPE
from sandbox.projects.yabs.qa.constants import NULL_STUB_SERVICES_SET, IGNORED_CACHEDEMON_STUBS


logger = logging.getLogger(__name__)


CLUSTER_CONFIG_PARAMETER = 'cluster_set_config'
CLUSTER_TAG_CONFIG_PARAMETER = 'cluster_tag'
META_MODE_TO_CLUSTER = {
    'bs': 'bs',
    'bsrank': 'yabs',
    'yabs': 'yabs',
}
META_MODE_TO_CLUSTER_TAG = {
    'bs': 109314048,
    'bsrank': 109379584,
    'yabs': 109445120,
}


class YabsSeparateAdapterBase(object):

    __metaclass__ = ABCMeta

    @staticmethod
    def dict_to_env(dict_object):
        return '\n'.join('{}={}'.format(name, str(value)) for name, value in dict_object.iteritems())

    def get_meta_mode(self):
        return self.parameters.meta_mode

    @abstractmethod
    def get_server_resource(self):
        pass

    @abstractmethod
    def get_binary_base_resources(self):
        pass

    # TODO: after separate adapters everywhere (delete YabsMetaStat...Adapters), need refactoring using this
    # @abstractmethod
    # def get_store_access_log(self):
    #     pass

    # @abstractmethod
    # def get_store_request_log(self):
    #     pass

    # @abstractmethod
    # def get_server_custom_environment(self):
    #     pass

    # def get_custom_environment(self):
    #     dict_environment = {
    #         CLUSTER_CONFIG_PARAMETER: self.get_cluster_config(),
    #         CLUSTER_TAG_CONFIG_PARAMETER: self.get_cluster_tag(),
    #     }
    #     dict_environment.update(self.get_server_custom_environment())
    #     return self.dict_to_env(dict_environment)

    def get_server_resource_path(self):
        return self.sync_resource(self.get_server_resource())

    def get_surf_data_resource_path(self):
        return self.sync_resource(self.parameters.surf_data_resource)

    def get_cluster_config(self):
        if self.parameters.specify_cluster_set_config:
            return META_MODE_TO_CLUSTER[self.get_meta_mode()]
        return 'all'

    def get_cluster_tag(self):
        if self.parameters.specify_cluster_tag:
            return META_MODE_TO_CLUSTER_TAG[self.get_meta_mode()]
        return 0

    def get_use_packed_base_by_tag(self, tag):
        return self.parameters.use_packed_bases

    def get_cache_daemon_stub_path(self):
        return self.sync_resource(self.parameters.cache_daemon_stub_resource)

    def create_cachedaemon(self, cachedaemon_data_dir=None, threads_per_service=None):
        cachedaemon_data_dir = cachedaemon_data_dir or get_unique_file_name(self.get_work_dir(), 'cache_daemon_data')
        cachedaemon_logs_dir = get_unique_file_name(self.get_logs_dir(), 'cache_daemon_logs')

        skip_services = NULL_STUB_SERVICES_SET | IGNORED_CACHEDEMON_STUBS
        if hasattr(self.parameters, 'use_separate_linear_models_service') and self.parameters.use_separate_linear_models_service:
            skip_services.add('linear_models_service')
            skip_services.add('linear_models_service_wide')

        return CacheDaemonStubSandboxNonIntegrated(
            cache_daemon_executable_path=self.get_cache_daemon_executable_path(),
            dump_path=self.get_cache_daemon_stub_path(),
            data_dir=cachedaemon_data_dir,
            log_subdir=cachedaemon_logs_dir,
            start_on_creation=False,
            services=SERVICES_BY_TYPE,
            skip_services=skip_services,
            threads_per_service=threads_per_service,
        )

    def get_server_path(self):
        destination_path = os.path.abspath(os.path.join(self.get_work_dir(), "yabs_server_bundle"))

        if os.path.isdir(destination_path):
            logger.debug("yabs-server bundle path exists: %s", destination_path)
            return destination_path

        server_resource_path = self.get_server_resource_path()
        logger.debug("Extract yabs-server bundle from %s to %s", server_resource_path, destination_path)
        with tarfile.open(server_resource_path) as archive:
            archive.extractall(path=destination_path)
        return destination_path

    def get_saas_frozen_topology(self):
        if self.parameters.saas_frozen_topology is None:
            return "\"\""

        path = self.sync_resource(self.parameters.saas_frozen_topology)
        return "\"{}\"".format(path)


class YabsAdapterBase(object):

    __metaclass__ = ABCMeta

    @abstractmethod
    def get_work_dir(self):
        pass

    @abstractmethod
    def get_logs_dir(self):
        pass

    @abstractmethod
    def get_base_dir(self):
        pass

    def get_binary_base_by_tag_dict(self):
        return self._binary_base_by_tag_dict

    @abstractmethod
    def get_base_resource_path_by_tag(self, tag):
        pass

    @abstractmethod
    def get_base_identity_key_by_tag(self, tag):
        pass

    @abstractmethod
    def get_transport_resource_path(self):
        pass

    @abstractmethod
    def get_cache_daemon_executable_path(self):
        pass
