import json
import logging

from sandbox.projects.yabs.qa.hamster.utils import calculate_enabled_hamsters
from sandbox.projects.yabs.qa.sut.metastat import YabsServerSeparateMetaSUT
from sandbox.projects.yabs.qa.sut.metastat.adapters.common import (
    CLUSTER_CONFIG_PARAMETER,
    CLUSTER_TAG_CONFIG_PARAMETER,
    YabsSeparateAdapterBase,
)
from sandbox.projects.yabs.qa.utils.resource import get_resource_id


logger = logging.getLogger(__name__)


class YabsMetaAdapter(YabsSeparateAdapterBase):
    def get_server_resource(self):
        return self.parameters.meta_server_resource

    def get_binary_base_resources(self):
        return self.parameters.meta_binary_base_resources

    def get_meta_store_access_log(self):
        return self.parameters.meta_store_access_log

    def get_meta_store_request_log(self):
        return self.parameters.meta_store_request_log

    def get_meta_custom_environment(self):
        dict_environment = {
            CLUSTER_CONFIG_PARAMETER: self.get_cluster_config(),
            CLUSTER_TAG_CONFIG_PARAMETER: self.get_cluster_tag(),
            'saas_static_topology_path': self.get_saas_frozen_topology()
        }

        saas_freeze_data = self.get_saas_freeze_data()

        logger.debug('SaaS freeze data: %s', saas_freeze_data)
        if saas_freeze_data is not None:
            snapshot_data = saas_freeze_data[0]  # we assume that there is exactly one snapshot
            dict_environment.update({
                'saas_pin_id': json.dumps(snapshot_data['id']),
                'saas_pin_stream': json.dumps(snapshot_data['stream']),
                'saas_pin_endpoint_set_id': json.dumps(snapshot_data['endpoint_set_id']),
                'saas_pin_endpoint_set_cluster': json.dumps(snapshot_data['endpoint_set_cluster']),
            })

        tags_using_sd_manager = []

        enabled_hamsters = calculate_enabled_hamsters(
            self.parameters.hamster_ext_service_tags or [],
            [get_resource_id(resource) for resource in (self.parameters.ext_service_endpoint_resources or [])],
        )
        logger.debug("Enabled hamsters: %s", enabled_hamsters)
        for service_tag, resource_id in enabled_hamsters.items():
            endpoint_spec = self.get_ext_service_endpoint_spec(resource_id)
            dict_environment.update({
                '{}_cluster'.format(service_tag): json.dumps(endpoint_spec['cluster']),
                '{}_endpoint_set'.format(service_tag): json.dumps(endpoint_spec['endpoint_set']),
            })
            if "meta_env" in endpoint_spec:
                dict_environment.update({
                    key: '"{}"'.format(value)
                    for key, value in endpoint_spec["meta_env"].items()
                })

            tags_using_sd_manager.append(json.dumps(service_tag))

        if tags_using_sd_manager:
            dict_environment.update({
                'tags_using_sd_manager': "{{ {} }}".format(' '.join(tags_using_sd_manager))
            })
        dict_environment.update(self.parameters.meta_custom_environment)
        return self.dict_to_env(dict_environment)

    def get_linear_models_binary_resource_path(self):
        return self.sync_resource(self.parameters.linear_models_binary_resource)

    def get_linear_models_data_resource_path(self):
        return self.sync_resource(self.parameters.linear_models_data_resource)

    def get_saas_freeze_data(self):
        if self.parameters.saas_freeze_data_resource is None:
            return None

        path = self.sync_resource(self.parameters.saas_freeze_data_resource)
        with open(path) as f:
            freeze_data = json.load(f)
        return freeze_data

    def get_ext_service_endpoint_spec(self, resource_id):
        path = self.sync_resource(resource_id)
        with open(path) as f:
            endpoint_spec = json.load(f)
        return endpoint_spec

    def create_module(self, *args, **kwargs):
        return YabsServerSeparateMetaSUT(self, *args, **kwargs)
