import logging
import os

from sandbox import sdk2
from sandbox.sandboxsdk.paths import make_folder, get_logs_folder

from sandbox.projects.resource_types import CACHE_DAEMON
from sandbox.projects.yabs.qa.adapter_base.sandbox import SandboxAdapterBase
from sandbox.projects.yabs.qa.sut.metastat.adapters.common import YabsAdapterBase
from sandbox.projects.yabs.qa.sut.metastat.adapters.common.meta import YabsMetaAdapter
from sandbox.projects.yabs.qa.sut.metastat.adapters.common.stat import YabsStatAdapter
from sandbox.projects.yabs.qa.sut.metastat.adapters.sandbox.parameters import (
    YabsMetaParameters,
    YabsMetaParameters2On1,
    YabsStatParameters,
    YabsStatParameters2On1
)
from sandbox.projects.yabs.qa.utils.base import get_transport_resource_path, BaseIdentityKey


logger = logging.getLogger(__name__)


class YabsSandboxAdapter(YabsAdapterBase, SandboxAdapterBase):
    def __init__(self, parameters, task_instance, work_dir=os.curdir):
        SandboxAdapterBase.__init__(self, parameters, task_instance)
        self._work_dir = work_dir
        self.build_binary_base_by_tag_dict()

    def build_binary_base_by_tag_dict(self):
        self._binary_base_by_tag_dict = {}
        for base in self.get_binary_base_resources():
            if isinstance(base, int):
                base = sdk2.Resource[base]
            if base.tag in self._binary_base_by_tag_dict:
                raise ValueError('Two or more bases with tag {} provided'.format(base.tag))
            self._binary_base_by_tag_dict[base.tag] = base

    def get_work_dir(self):
        return self._work_dir

    def get_logs_dir(self):
        return get_logs_folder()

    def get_base_dir(self):
        if self.parameters.use_tmpfs:
            return self.task_instance.ramdrive.path
        base_path = os.path.abspath('yabs_server_bases')
        make_folder(base_path)
        return base_path

    def get_base_resource_path_by_tag(self, tag):
        return self.sync_resource(self._binary_base_by_tag_dict[tag])

    def get_base_identity_key_by_tag(self, tag):
        return BaseIdentityKey(
            resource_id=self._binary_base_by_tag_dict[tag].id,
            use_packed=self.parameters.use_packed_bases
        )

    def get_transport_resource_path(self):
        return get_transport_resource_path()

    def get_cache_daemon_executable_path(self):
        cache_daemon_executable_res_id = self.parameters.cache_daemon_stub_resource.cache_daemon_res_id
        return self.sync_resource(sdk2.Resource.find(CACHE_DAEMON, id=cache_daemon_executable_res_id).first())


class YabsMetaSandboxAdapter(YabsMetaAdapter, YabsSandboxAdapter):
    @staticmethod
    def get_init_parameters_class(for_2_on_1=False):
        return YabsMetaParameters2On1 if for_2_on_1 else YabsMetaParameters


class YabsStatSandboxAdapter(YabsStatAdapter, YabsSandboxAdapter):
    @staticmethod
    def get_init_parameters_class(for_2_on_1=False):
        return YabsStatParameters2On1 if for_2_on_1 else YabsStatParameters
