import json
import logging
import os

from sandbox.common.rest import Client

from sandbox.projects.yabs.qa.adapter_base.standalone import StandaloneAdapterBase
from sandbox.projects.yabs.qa.sut.metastat.adapters.common import YabsAdapterBase
from sandbox.projects.yabs.qa.sut.metastat.adapters.common.meta import YabsMetaAdapter
from sandbox.projects.yabs.qa.sut.metastat.adapters.common.stat import YabsStatAdapter
from sandbox.projects.yabs.qa.utils.base import BaseIdentityKey


logger = logging.getLogger(__name__)


class YabsStandaloneAdapter(YabsAdapterBase, StandaloneAdapterBase):
    def __init__(
            self,
            parameters,
            logs_dir='logs',
            resource_cache_dir='resource_cache',
            work_dir='work_dir',
            base_dir='base_dir',
    ):
        StandaloneAdapterBase.__init__(self, parameters, logs_dir, resource_cache_dir)
        self._client = Client()
        self.build_binary_base_by_tag_dict()

        self._work_dir = work_dir
        if not os.path.isdir(self._work_dir):
            os.makedirs(self._work_dir)

        self._base_dir = base_dir
        if not os.path.isdir(self._base_dir):
            os.makedirs(self._base_dir)

    def build_binary_base_by_tag_dict(self):
        logger.debug('Starting binary base metadata pre-fetch')
        self._binary_base_by_tag_dict = {}
        for base_resource_id in self.get_binary_base_resources():
            base_resource_attrs = self.get_resource_attributes(base_resource_id)
            base_tag = base_resource_attrs['tag']
            if base_tag in self._binary_base_by_tag_dict:
                raise ValueError('Two or more bases with tag {} provided'.format(base_tag))
            self._binary_base_by_tag_dict[base_tag] = {
                'id': base_resource_id,
                'attributes': base_resource_attrs
            }
        logger.debug('Binary base metadata pre-fetch complete')

    def get_work_dir(self):
        return self._work_dir

    def get_logs_dir(self):
        return self._logs_dir

    def get_base_dir(self):
        return self._base_dir

    def get_base_resource_path_by_tag(self, tag):
        return self.sync_resource(self._binary_base_by_tag_dict[tag]['id'])

    def get_base_identity_key_by_tag(self, tag):
        return BaseIdentityKey(
            resource_id=self._binary_base_by_tag_dict[tag]['id'],
            use_packed=self.parameters.use_packed_bases
        )

    def get_transport_resource_path(self):
        transport_resource_data = self._client.resource.read(limit=100,
                                                             order='-id',
                                                             type='YABS_SERVER_BASE_PACK_EXECUTABLE',
                                                             attrs=json.dumps({'released': 'stable'}))['items'][0]
        return os.path.join(self.sync_resource(transport_resource_data['id']), 'transport')

    def get_cache_daemon_executable_path(self):
        cache_daemon_stub_resource_data = self._client.resource[self.parameters.cache_daemon_stub_resource].read()
        cache_daemon_executable_res_id = cache_daemon_stub_resource_data['attributes']['cache_daemon_res_id']
        return self.sync_resource(cache_daemon_executable_res_id)


class YabsMetaStandaloneAdapter(YabsMetaAdapter, YabsStandaloneAdapter):
    pass


class YabsStatStandaloneAdapter(YabsStatAdapter, YabsStandaloneAdapter):
    pass
