import logging
import itertools
import os
import re
import subprocess
from abc import ABCMeta, abstractmethod

from sandbox.projects.yabs.qa.module_base import ModuleBase
from sandbox.projects.yabs.qa.sut.factory import YabsServerFactoryStandalone
from sandbox.projects.yabs.qa.sut.components.cachedaemon_adapter import YabsCacheDaemonStubAdapter
from sandbox.projects.yabs.qa.sut.bases_provider.adapter_compatible import BinBasesProvider
from sandbox.projects.yabs.qa.sut.utils import ListContext
from sandbox.projects.yabs.qa.constants import NULL_STUB_SERVICES_SET

COOKIE_PATTERN = re.compile(r'(?P<name>[a-zA-Z_-]*)=(?P<value>.*)')


class YabsServerSeparateSUTBase(ModuleBase):

    __metaclass__ = ABCMeta

    class UsageDataContext(object):
        def __init__(self, module):
            self.__module = module

        def __enter__(self):
            self.__usage_data_dict = {'start': self.__module.get_server_backend_object().get_usage_data()}
            return self.__usage_data_dict

        def __exit__(self, *args):
            self.__usage_data_dict['stop'] = self.__module.get_server_backend_object().get_usage_data()

    def get_usage_data(self):
        return self.UsageDataContext(self)

    FACTORY_TYPE = YabsServerFactoryStandalone

    def __init__(
        self,
        adapter,
        cachedaemon,
        shared_base_state,
        maximum_keys_per_tag,
        wait_ping_timeout,
        yabs_server_bundle_path=None,
    ):
        ModuleBase.__init__(self, adapter)

        self._wait_ping_timeout = wait_ping_timeout

        self._cachedaemon = cachedaemon
        self._cachedaemon_adapter = YabsCacheDaemonStubAdapter(self._cachedaemon)

        self._server_path = yabs_server_bundle_path
        if not self._server_path:
            self._server_path = self.adapter.get_server_path()

        self._factory = self.FACTORY_TYPE(
            server_path=self._server_path,
            surf_data_path=self.adapter.get_surf_data_resource_path(),
            task_instance=self.adapter.task_instance,
            work_dir=self.adapter.get_work_dir(),
            logs_dir=self.adapter.get_logs_dir(),
            base_dir=self.adapter.get_base_dir(),
            prefix=self.__class__.__name__,
        )
        self._null_stub = self._factory.create_null_stub(provided_services=NULL_STUB_SERVICES_SET)

        self._base_provider = BinBasesProvider(self.adapter, shared_base_state, maximum_keys_per_tag)

    @property
    def shared_base_state(self):
        return self._base_provider.base_state

    @property
    def base_provider(self):
        return self._base_provider

    @property
    def base_revision(self):
        return self._factory.server_base_revision

    @property
    def cachedaemon(self):
        return self._cachedaemon

    def get_debug_cookies(self):
        cmd = [
            os.path.join(self._factory.server_path, 'yabs_debug_cookie'),
            '--bases',
            str(self.adapter.get_base_dir()),
        ]
        logging.debug('Launching `%s`', ' '.join(cmd))
        raw_cookies = subprocess.check_output(cmd).split('\n')
        logging.debug('Got raw cookies: %s', raw_cookies)
        cookies = []
        for raw_cookie in raw_cookies:
            if not raw_cookie:
                continue
            try:
                m = COOKIE_PATTERN.match(raw_cookie)
                if not m:
                    logging.debug('Cannot extract cookie from %s', raw_cookie)
                    continue
                cookie = m.groupdict()['value']
            except Exception:
                logging.warning('Cannot extract cookie from %s', raw_cookie, exc_info=True)
            else:
                cookies.append(cookie)
        return cookies

    @abstractmethod
    def _get_server_iterable(self):
        pass

    @abstractmethod
    def _get_wait_ping_iterable(self):
        pass

    def __enter__(self):
        logging.debug("ENTER %s", self.__class__.__name__)
        self._server_context = ListContext(self._get_server_iterable())
        self._server_context.__enter__()
        wait_ping_iterable = self._get_wait_ping_iterable()
        for instance in wait_ping_iterable:
            instance.wait_ping(timeout=self._wait_ping_timeout)
        logging.debug("%s started", self.__class__.__name__)
        return self

    def __exit__(self, *args):
        logging.debug("EXIT %s", self.__class__.__name__)
        self._server_context.__exit__(*args)
        self._server_context = None
        self._server_iterable = None
        self._server_backend_no = 0
        logging.debug("%s shut down", self.__class__.__name__)

    @abstractmethod
    def get_server_backend_object(self):
        pass

    def get_port(self):
        if self._server_context:
            return self.get_server_backend_object().listen_port
        else:
            return None

    def is_active(self):
        return all(server.process.poll() is None for server in self._get_server_iterable())

    @abstractmethod
    def _get_backends_iter(self):
        pass

    def get_ext_tag_set(self):
        backends_iter = self._get_backends_iter()
        tag_set = set(itertools.chain.from_iterable(b.get_provided_ext_tags() for b in backends_iter))
        if 'bscount' in tag_set:
            tag_set.add('bscount_proto')
            tag_set.add('bscount_proto2')
            tag_set.add('bscount_fbs')
            tag_set.add('bscount_fbs2')
        return tag_set
