import json
import logging
import os
import re

from collections import Counter

from enum import Enum


LOGGER = logging.getLogger(__name__)
MISS_STATUSES = ('fails', 'miss')
HIT_STATUSES = ('hit', 'success')
SAAS_SENSOR_PATTERN = re.compile(r'(?P<stream>[a-z0-9_]+)_(?P<status>{status})'.format(status='|'.join(HIT_STATUSES + MISS_STATUSES)))


# https://a.yandex-team.ru/arc_vcs/yabs/server/libs/server/solomon/types.h?rev=3519c760a89d8f24197b0c080c33433d1f9d1d1a#L14
class SolomonStatsServices(Enum):
    adfox = 1
    base_info = 2
    degradation_per_host = 3
    events = 4
    ext_mirror_requests_per_host = 5
    ext_mirror_requests = 6
    ext_requests_per_host = 7
    ext_requests = 8
    external_sensors = 9
    filtrations = 10
    formula = 11
    lm = 12
    other = 13
    pmatch = 14
    requests = 15
    saas_stat_per_host = 16
    server_side_cache = 17
    service_discovery = 18
    stat_load = 19
    stat_sensors = 20
    timedout_stages = 21
    timings_access_stat_log = 22
    timings_hit_log = 23
    unified_agent_per_host = 24
    unified_agent = 25
    uptime = 26
    validator = 27


def get_solomon_stats(port, service):
    import requests

    url = "http://localhost:{}/solomon_stats".format(port)
    try:
        response = requests.get(url, params={"service": service})
        response.raise_for_status()
    except Exception as e:
        LOGGER.error(e)
        return None

    try:
        return response.json()
    except json.JSONDecodeError as e:
        LOGGER.error("Failed to decode JSON. %s. Response: \"%s\"", e, response.text)
        return None


def get_file_name(module_name, service):
    return "{module_name}_{service}.json".format(module_name=module_name, service=service)


def dump_solomon_stats(server_backend, out_dir):
    for service in SolomonStatsServices:
        solomon_stats_data = get_solomon_stats(server_backend.solomon_stats_port, service.name)
        if not solomon_stats_data:
            continue
        solomon_stats_filename = get_file_name(server_backend.name, service.name)
        solomon_stats_file_path = os.path.join(out_dir, solomon_stats_filename)
        with open(solomon_stats_file_path, "w") as f:
            json.dump(solomon_stats_data, f)
        LOGGER.debug("Dump solomon_stats to %s", solomon_stats_file_path)


def generate_key(labels_dict, ignore_labels=("sensor", "role", "server_group")):
    return tuple(
        labels_dict[k]
        for k in sorted(set(labels_dict.keys()) - set(ignore_labels))
    )


def validate_saas_solomon_stats(module_name, solomon_stats_dir, miss_threshold=0.05):
    service = SolomonStatsServices.saas_stat_per_host
    solomon_stats_filename = get_file_name(module_name, service.name)
    solomon_stats_file_path = os.path.join(solomon_stats_dir, solomon_stats_filename)
    with open(solomon_stats_file_path, 'r') as solomon_stats_file:
        sensors = json.load(solomon_stats_file)["sensors"]
    counters = {}
    for sensor in sensors:
        labels = sensor["labels"]
        m = SAAS_SENSOR_PATTERN.search(labels["sensor"])
        if not m:
            LOGGER.debug('Sensor name in %s does not match pattern <stream>_(hit|success|miss|fails), skipping', labels)
            continue
        if labels.get('type'):
            LOGGER.debug('Label \'type\' in %s is set, skipping', labels)
            continue
        key = (m.group('stream'), ) + generate_key(labels)
        _type = "hit" if m.group('status') in HIT_STATUSES else "miss"
        counters.setdefault(key, Counter())[_type] += sensor["value"]

    LOGGER.debug("Count results: %s", counters)

    failures = []
    for key, stat in counters.items():
        if stat["hit"] + stat["miss"] > 0 and float(stat["miss"]) / (stat["hit"] + stat["miss"]) > miss_threshold:
            failures.append((key, stat["hit"], stat["miss"]))
    return failures


def create_saas_solomon_stats_validation_report(module_name, failures):
    return "=== {module_name} SaaS solomon_stats validation {status} ===\n{failures}".format(
        module_name=module_name,
        status='failed' if failures else 'passed',
        failures='\n'.join([
            '{key} validation failed, {miss_percent:.2f}% misses, {miss} out of {total}'.format(
                key='.'.join(key),
                miss_percent=100 * float(miss) / (hit + miss),
                miss=miss,
                total=hit + miss,
            )
            for key, hit, miss in failures
        ]),
    )
