import os
import logging

from sandbox import sdk2
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.projects.yabs.qa.mutable_parameters import MutableParameters
from sandbox.projects.yabs.base_bin_task import BaseBinTask


def factory(
    sut_module_adapter_class,
    ammo_module_adapter_class,
    shoot_module_adapter_class,
    session_aggr_module_adapter_class,
    dump_resource_class,
    name_postfix,
    is_2on1,
):
    class TaskCommonParameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('General settings') as general_parameters:
            shoot_sessions = sdk2.parameters.Integer('Shoot sessions', default_value=3)
            add_readable_results = sdk2.parameters.Bool('Print readable results', default=False)
            flush_state = sdk2.parameters.Bool('Flush service state after shoot sessions', default=False)

            with sdk2.parameters.Output:
                with add_readable_results.value[True]:
                    readable_results = sdk2.parameters.JSON('Perf readable results', required=True)
                results_resource = sdk2.parameters.Resource('Perf results resource', required=True)

        ammo_module_parameters = ammo_module_adapter_class.get_init_parameters_class()()
        shoot_module_parameters = shoot_module_adapter_class.get_init_parameters_class()()
        session_aggr_module_parameters = session_aggr_module_adapter_class.get_init_parameters_class()()

    class TaskSeparateParameters(sdk2.Task.Parameters):
        sut_module_parameters = sut_module_adapter_class.get_init_parameters_class()()
        if is_2on1:
            with sdk2.parameters.Group('2on1 parameters') as params_2on1:
                second_run_parameters = sut_module_adapter_class.get_init_parameters_class(second_run=True)()

    class TaskParameters(sdk2.Task.Parameters):
        common_parameters = TaskCommonParameters()
        separate_parameters = TaskSeparateParameters()

    class TaskClass(BaseBinTask):
        name = 'FACTORY_SHOOT_{}_{}'.format(name_postfix, '2_ON_1' if is_2on1 else '1_ON_1')

        CommonParameters = TaskCommonParameters
        SeparateParameters = TaskSeparateParameters
        Parameters = TaskParameters

        def run_pipeline(self, parameters, index_2on1=1):
            work_dir = 'sut_adapter_{}'.format(index_2on1)
            if os.path.exists(work_dir):
                raise SandboxTaskFailureError('Directory %s already exists' % work_dir)

            sut_service = sut_module_adapter_class(parameters, self, work_dir=work_dir).create_module()
            ammo_module = ammo_module_adapter_class(parameters, self).create_module()
            shoot_module = shoot_module_adapter_class(parameters, self).create_module()

            for i in range(parameters.shoot_sessions):
                logging.debug('Starting session %s, 2on1 index: %s', i, index_2on1)

                with sut_service as active_service:
                    dump_path = shoot_module.shoot_and_watch(active_service, ammo_module.get_dplan_path())
                    logging.debug('Saved dump to %s', dump_path)
                self.session_aggr_module.feed_dump(dump_path, index=index_2on1)
                os.remove(dump_path)

            if parameters.flush_state:
                sut_service.flush_state()

        def generate_second_run_parameters(self, parameters):
            second_run_parameters = MutableParameters.__from_parameters__(parameters)
            for name, value in parameters:
                if name.endswith('_2'):
                    setattr(second_run_parameters, name[:-2], value)
            return second_run_parameters

        def on_execute(self):
            self.session_aggr_module = session_aggr_module_adapter_class(self.Parameters, self, dump_resource_class=dump_resource_class).create_module()

            parameters_list = [
                self.Parameters,
            ]

            if is_2on1:
                test_parameters = self.generate_second_run_parameters(self.Parameters)
                parameters_list.append(test_parameters)

            for index, parameters in enumerate(parameters_list):
                self.run_pipeline(
                    parameters=parameters,
                    index_2on1=index+1,
                )

            with self.memoize_stage.set_output:
                self.Parameters.results_resource = self.session_aggr_module.upload_results()
                if self.Parameters.add_readable_results:
                    self.Parameters.readable_results = self.session_aggr_module.get_readable_results()

    return TaskClass
