import datetime
import logging
import uuid

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.yabs.qa.utils import yt_utils
from sandbox.sandboxsdk import environments


logger = logging.getLogger(__name__)

DEFAULT_BACKUP_DESTINATION_PATH = '//home/yabs-cs-sandbox/input-archive-oneshots'
DEFAULT_BACKUP_TTL = datetime.timedelta(hours=1)
DEFAULT_YT_TOKEN_VAULT_NAME = 'yabs-cs-sb-yt-token'
DEFAULT_YT_PROXY = 'hahn'


def dump_yt_table(yt_client, src, dst):
    """
    https://wiki.yandex-team.ru/yt/userdoc/dynamictablesmapreduce/#preobrazovaniedinamicheskojjtablicyvstaticheskujuiobratno

    """
    import yt

    with yt_client.Transaction():
        # Take snapshot lock for source table.
        yt_client.lock(src, mode="snapshot")
        # Create table and save vital attributes.
        yt_client.create_table(
            dst,
            attributes={
                "optimize_for": yt_client.get_attribute(src, "optimize_for", default="lookup"),
                "schema": yt_client.get_attribute(src, "schema"),
            },
            recursive=True,
        )
        # Get timestamp of flushed data.
        # May be None if table is ordered https://wiki.yandex-team.ru/yt/userdoc/dynamicorderedtables/
        unflushed_timestamp = yt_client.get_attribute(src, "unflushed_timestamp", default=None)
        attributes = {}
        if unflushed_timestamp is not None:
            attributes["timestamp"] = unflushed_timestamp - 1
        # Dump table contents
        yt_client.run_merge(yt.wrapper.table.TablePath(src, attributes=attributes), dst, mode="ordered")


class BackupYTTables(sdk2.Task):
    """Backup YT tables"""

    name = 'BACKUP_YT_TABLES'

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024
        environments = (
            environments.PipEnvironment('yandex-yt', use_wheel=True),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        tables = sdk2.parameters.List('Tables', required=True)
        destination_path = sdk2.parameters.String(
            'Destination path',
            description=(
                'Destination path for backup tables. By default new node will be created in {}'
                .format(DEFAULT_BACKUP_DESTINATION_PATH)
            )
        )
        yt_proxy = sdk2.parameters.String('YT proxy', default='hahn', required=True)
        backup_dir_ttl = sdk2.parameters.Integer(
            'TTL',
            description='TTL for backups in seconds',
            required=True,
            default=DEFAULT_BACKUP_TTL.total_seconds()
        )
        yt_token_vault_name = sdk2.parameters.String('Vault name for YT token', default=DEFAULT_YT_TOKEN_VAULT_NAME)

        with sdk2.parameters.Output:
            backup_dir = sdk2.parameters.String('Backup directory',
                                                description='Tables would be copied to this directory')

    def on_execute(self):
        from yt.wrapper import YtClient, ypath_join

        yt_client = YtClient(proxy=self.Parameters.yt_proxy, token=self.yt_token)

        tables = set(self.Parameters.tables)
        for src_table_path in tables:
            if not yt_client.exists(src_table_path):
                raise TaskFailure('Table "{}" not found'.format(src_table_path))

            if yt_client.get_attribute(src_table_path, 'type') != 'table':
                raise TaskFailure('"{}" is not a "table"'.format(src_table_path))

        backup_dir = self.Parameters.destination_path
        if not backup_dir:
            backup_dir = ypath_join(DEFAULT_BACKUP_DESTINATION_PATH, str(uuid.uuid4()))

            logger.info('Create table "%s"', backup_dir)
            yt_client.create('map_node', backup_dir)

            logger.info('Set TTL=%s seconds for table "%s"', self.Parameters.backup_dir_ttl, backup_dir)
            yt_utils.set_yt_node_ttl(backup_dir, self.Parameters.backup_dir_ttl, yt_client)

        logger.info('Backup directory: %s', backup_dir)

        for src_table_path in tables:
            dst_table_path = ypath_join(backup_dir, src_table_path.lstrip('/'))
            self.copy_table(src_table_path, dst_table_path)

        logger.info('Copy completed')
        self.Parameters.backup_dir = backup_dir

    @property
    def yt_token(self):
        return sdk2.Vault.data(self.Parameters.yt_token_vault_name)

    def copy_table(self, src_path, dst_path, client=None):
        if client is None:
            from yt.wrapper import YtClient
            client = YtClient(proxy=self.Parameters.yt_proxy, token=self.yt_token)

        logger.info('Copy table "%s" to "%s"', src_path, dst_path)

        if not client.get_attribute(src_path, 'dynamic', default=False):
            logger.info('Source table "%s" is "static"', src_path)
            client.copy(src_path, dst_path, recursive=True)
            return
        logger.info('Source table "%s" is "dynamic"', src_path)

        current_src_table_state = client.get_attribute(src_path, 'tablet_state')
        logger.info('Source table "%s" is "%s"', src_path, current_src_table_state)

        dump_yt_table(client, src_path, dst_path)

        logger.info('Convert destination table "%s" to "dynamic"', dst_path)
        client.alter_table(dst_path, dynamic=True)

        logger.info('Set destination table "%s" state to "%s"', dst_path, current_src_table_state)
        set_table_state(client, dst_path, current_src_table_state)


def set_table_state(yt_client, table_path, state):
    if not yt_client.get_attribute(table_path, 'dynamic', default=False):
        return

    current_state = yt_client.get_attribute(table_path, 'tablet_state')
    if current_state == state:
        return

    if current_state == 'unmounted':
        yt_client.mount_table(table_path, sync=True)

        if state == 'mounted':
            return

        yt_client.freeze_table(table_path, sync=True)
        return

    if current_state == 'mounted':
        if state == 'unmounted':
            yt_client.unmount_table(table_path, sync=True)
            return

        yt_client.freeze_table(table_path, sync=True)
        return

    if current_state == 'frozen':
        yt_client.unfreeze_table(table_path, sync=True)

        if state == 'mounted':
            return

        yt_client.unmount_table(table_path, sync=True)
        return
